unit Common.Config;

interface

const
  defaultServerUrl = 'http://localhost:2004/kgOrders/';

type
  TServerConfig = class
  private
    Furl: string;
    FjwtTokenSecret: string;
    FadminPassword: string;
    FwebAppFolder: string;
    FreportsFolder: string;
  public
    property url: string read Furl write Furl;
    property jwtTokenSecret: string read FjwtTokenSecret write FjwtTokenSecret;
    property adminPassword: string read FadminPassword write FadminPassword;
    property webAppFolder: string read FwebAppFolder write FwebAppFolder;
    property reportsFolder: string read FreportsFolder write FreportsFolder;

    constructor Create;
  end;

procedure LoadServerConfig;

var
  serverConfig: TServerConfig;

implementation

uses
  Bcl.Json, System.SysUtils, System.IOUtils, Common.Logging, System.StrUtils;

procedure LoadServerConfig;
var
  configFile: string;
  localConfig: TServerConfig;
begin
  Logger.Log(1, '--LoadServerConfig - start');
  configFile := TPath.ChangeExtension(ParamStr(0), '.json');
  Logger.Log(1, '-- Config file: ' + ConfigFile);

  if TFile.Exists(configFile) then
  begin
    Logger.Log(1, '-- Config file found.');
    localConfig := TJson.Deserialize<TServerConfig>(TFile.ReadAllText(configFile));
    Logger.Log(1, '-- localConfig loaded from config file');

    serverConfig.Free;
    Logger.Log(1, '-- serverConfig.Free - called');
    serverConfig := localConfig;
    Logger.Log(1, '-- serverConfig := localConfig - called');
  end
  else
  begin
    Logger.Log(1, '-- Config file not found.');
  end;

  Logger.Log(1, '--- Server Config Values ---');
  Logger.Log(1, '-- url: ' + serverConfig.url + IfThen(serverConfig.url = defaultServerUrl, ' [default]', ' [from config]'));
  Logger.Log(1, '-- adminPassword: ' + serverConfig.adminPassword + IfThen(serverConfig.adminPassword = 'whatisthisusedfor', ' [default]', ' [from config]'));
  Logger.Log(1, '-- jwtTokenSecret: ' + serverConfig.jwtTokenSecret + IfThen(serverConfig.jwtTokenSecret = 'super_secret0123super_secret4567', ' [default]', ' [from config]'));
  Logger.Log(1, '-- webAppFolder: ' + serverConfig.webAppFolder + IfThen(serverConfig.webAppFolder = 'static', ' [default]', ' [from config]'));
  Logger.Log(1, '-- serverConfig.reportsFolder: ' + serverConfig.reportsFolder);

  if not DirectoryExists(serverConfig.reportsFolder) then
  begin
    ForceDirectories(serverConfig.reportsFolder);
    Logger.Log(1, '-- Reports directory created: ' + serverConfig.reportsFolder);
  end;

  Logger.Log(1, '--LoadServerConfig - end');
end;

{ TServerConfig }

constructor TServerConfig.Create;
var
  ServerConfigStr: string;
begin
  Logger.Log(1, '--TServerConfig.Create - start');
  url := defaultServerUrl;
  adminPassword := 'whatisthisusedfor';
  jwtTokenSecret := 'super_secret0123super_secret4567';
  webAppFolder := 'static';
  reportsFolder := 'static\reports\';
  ServerConfigStr := Bcl.Json.TJson.Serialize(ServerConfig);
  Logger.Log(1, '--ServerConfigSerialize: ' + ServerConfigStr);
  Logger.Log(1, '--TServerConfig.Create - end');
end;


end.
