// Add Customer page for KGOrders. Handles Adding and Editting Customers and
// Their shipping addresses.
unit AddCustomer;

interface

uses
  System.SysUtils, System.Generics.Collections, System.Classes, JS, Web, WEBLib.Graphics, WEBLib.Controls,
  WEBLib.Forms, WEBLib.Dialogs, WEBLib.Menus, WEBLib.ExtCtrls, WEBLib.StdCtrls,
  WEBLib.JSON, Auth.Service, XData.Web.Client, WebLib.Storage,
  ConnectionModule, App.Types, Vcl.StdCtrls, Vcl.Controls, WEBLib.DBCtrls,
  Data.DB, XData.Web.JsonDataset, XData.Web.Dataset, WEBLib.DB, WEBLib.Grids, VCL.Forms;

type
  TFViewAddCustomer = class(TWebForm)
    dtpStartDate: TWebDateTimePicker;
    dtpEndDate: TWebDateTimePicker;
    edtShortName: TWebDBEdit;
    edtName: TWebDBEdit;
    edtBillAddress: TWebDBEdit;
    edtBillCity: TWebDBEdit;
    edtBillState: TWebDBEdit;
    edtBillZip: TWebDBEdit;
    edtBillContact: TWebDBEdit;
    edtPhone: TWebDBEdit;
    edtFax: TWebDBEdit;
    btnSave: TWebButton;
    btnCancel: TWebButton;
    btnDelete: TWebButton;
    btnClose: TWebButton;
    btnEdit: TWebButton;
    XDataWebClient1: TXDataWebClient;
    WebDataSource1: TWebDataSource;
    XDataWebDataSet1: TXDataWebDataSet;
    XDataWebDataSet1SHORT_NAME: TStringField;
    XDataWebDataSet1NAME: TStringField;
    XDataWebDataSet1BILL_ADDRESS: TStringField;
    XDataWebDataSet1BILL_CITY: TStringField;
    edtCustomerID: TWebDBEdit;
    edtQBID: TWebDBEdit;
    wdbtcAddresses: TWebDBTableControl;
    wdsShipTo: TWebDataSource;
    xdwdsShipTo: TXDataWebDataSet;
    XDataWebDataSet1BILL_STATE: TStringField;
    XDataWebDataSet1BILL_CONTACT: TStringField;
    XDataWebDataSet1BILL_ZIP: TStringField;
    XDataWebDataSet1START_DATE: TStringField;
    XDataWebDataSet1END_DATE: TStringField;
    XDataWebDataSet1QB_LIST_ID: TStringField;
    XDataWebDataSet1FAX: TStringField;
    XDataWebDataSet1PHONE: TStringField;
    XDataWebDataSet1CUSTOMER_ID: TIntegerField;
    edtShippingAddress: TWebEdit;
    edtShippingState: TWebEdit;
    edtShippingContact: TWebEdit;
    edtShippingZip: TWebEdit;
    edtShippingCity: TWebEdit;
    xdwdsShipToADDRESS: TStringField;
    xdwdsShipToship_id: TStringField;
    xdwdsShipToshipping_address: TStringField;
    xdwdsShipTocity: TStringField;
    xdwdsShipTostate: TStringField;
    xdwdsShipTozip: TStringField;
    xdwdsShipTocontact: TStringField;
    memoShipBlock: TWebMemo;
    memoAddressBlock: TWebMemo;
    btnAdd: TWebButton;
    btnShipSave: TWebButton;
    btnShipCancel: TWebButton;
    btnShipDelete: TWebButton;
    btnShipEdit: TWebButton;
    btnShipAdd: TWebButton;
    tmrReturn: TWebTimer;
    edtFirstLine: TWebEdit;
    wdsUsers: TWebDataSource;
    xdwdsUsers: TXDataWebDataSet;
    wdblcbRep: TWebDBLookupComboBox;
    xdwdsUsersuserID: TStringField;
    XDataWebDataSet1REP_USER_ID: TStringField;
    xdwdsUsersfull_name: TStringField;
    lblFormState: TWebLabel;
    [async]
    xdwdsUsersrepresentative: TStringField; procedure btnSaveClick(Sender: TObject);
    procedure btnCancelClick(Sender: TObject);
    procedure btnCloseClick(Sender: TObject);
    procedure btnEditClick(Sender: TObject);
    procedure wdbtcAddressesDblClickCell(Sender: TObject; ACol, ARow: Integer);
    procedure btnClearClick(Sender: TObject);
    procedure AddressEditMode();
    procedure edtShippingAddressChange(Sender: TObject);
    procedure btnAddClick(Sender: TObject);
    procedure tmrReturnTimer(Sender: TObject);
    procedure btnDeleteClick(Sender: TObject);
    procedure btnShipAddClick(Sender: TObject);
    procedure btnShipEditClick(Sender: TObject);
    [async] procedure btnShipSaveClick(Sender: TObject);
    procedure btnShipDeleteClick(Sender: TObject);
    procedure btnShipCancelClick(Sender: TObject);
  private
    { Private declarations }
    procedure ViewMode();
    procedure EditMode();
    [async] procedure GetCustomer();
    [async] procedure SendCustomerToServer();
    [async] procedure SendAddressToServer();
    [async] procedure DelAddress();
    [async] procedure Save();
    function VerifyCustomer(): boolean;
    function VerifyAddress(): boolean;
    procedure Clear();
    procedure ShowSelectCustomerForm();
    [async] procedure InitializeForm;
  var
    customerID: string;
    notification: string;
    mode: string;
    shipmode: string;
  public
    { Public declarations }
     class function CreateForm(AElementID, customerInfo, info: string): TWebForm;
  end;

var
  FViewAddCustomer: TFViewAddCustomer;

implementation

{$R *.dfm}

uses View.Main, View.Customers, View.SelectCustomer, Utils;


class function TFViewAddCustomer.CreateForm(AElementID, customerInfo, info: string): TWebForm;
begin
  Application.CreateForm(TFViewAddCustomer, AElementID, Result,
    procedure(AForm: TObject)
    begin
      with TFViewAddCustomer(AForm) do
      begin
        customerID := customerInfo;
        notification := info;

        InitializeForm;
      end;
    end
  );
end;


[async] procedure TFViewAddCustomer.InitializeForm;
begin
  if customerID = '' then
    mode := 'ADD'
  else
    mode := 'EDIT';

  if notification <> '' then
    ShowToast(notification);

  if mode = 'ADD' then
    EditMode()
  else
    ViewMode();

  await(GetCustomer);

  dtpStartDate.Date := 0;
  dtpEndDate.Date := 0;
end;

procedure TFViewAddCustomer.Clear();
// Clears the shipping address fields.
begin
  edtShippingAddress.Text := '';
  edtShippingCity.Text := '';
  edtShippingState.Text := '';
  edtShippingZip.Text := '';
  edtShippingContact.Text := '';
  memoShipBlock.Text := '';
  edtFirstLine.Text := '';
end;

procedure TFViewAddCustomer.ShowSelectCustomerForm();
// displays the add order pop-up so the user can choose a customer
var
  newform: TFSelectCustomer;
begin
  newform := TFSelectCustomer.CreateNew;

  newform.Caption := 'Select Customer and Order Type';
  newForm.Popup := True;
  newForm.Border := fbDialog;
  newForm.Position := poScreenCenter;

  // used to manage Back button handling to close subform
  window.location.hash := 'subform';

  newform.ShowModal(
    procedure(AValue: TModalResult)
    begin

    end
  );
end;

procedure TFViewAddCustomer.DelAddress;
// Deletes a shipping address.
var
  Response: TXDataClientResponse;
  notification: TJSObject;
begin
  Response := await(XDataWebClient1.RawInvokeAsync('ILookupService.DelShippingAddress',
              [xdwdsShipTo.FieldByName('ship_id').AsString, customerID]));
  notification := TJSObject(Response.Result);
  ShowToast(string(notification['status']));
  xdwdsShipTo.Close;
  xdwdsShipTo.SetJSONData(notification['ADDRESS']);
  xdwdsShipTo.Open;
  Utils.HideSpinner('spinner');
end;

procedure TFViewAddCustomer.SendAddressToServer;
// Creates an Address JSON and then sends it to the server for the address to be
// Added or edited.
var
  Field: TField;
  AddressJSON: TJSONObject;
  Response: TXDataClientResponse;
  notification: TJSObject;
  ship_block: string;
begin
  AddressJSON := TJSONObject.Create;

  AddressJSON.AddPair('address', edtShippingAddress.Text);
  AddressJSON.AddPair('city', edtShippingcity.Text);
  AddressJSON.AddPair('state', edtShippingstate.Text);
  AddressJSON.AddPair('zip', edtShippingzip.Text);
  AddressJSON.AddPair('contact', edtShippingContact.Text);
  AddressJSON.AddPair('customer_id', customerID);

  ship_block := edtFirstLine.Text + slinebreak +
                edtName.Text + slinebreak +
                edtShippingContact.Text + slinebreak +
                edtShippingAddress.Text + slinebreak +
                edtShippingCity.Text + ', ' + edtShippingState.Text + ' ' + edtShippingZip.Text;
  AddressJSON.AddPair('ship_block', ship_block);

  if shipmode = 'EDIT' then
    AddressJSON.AddPair('customer_ship_id', xdwdsShipTo.FieldByName('ship_id').AsString);

  AddressJSON.AddPair('mode', shipmode);

  Response := await(XDataWebClient1.RawInvokeAsync('ILookupService.AddShippingAddress',
              [AddressJSON.ToString]));

  notification := TJSObject(Response.Result);
  ShowToast(string(notification['status']));
  xdwdsShipTo.Close;
  xdwdsShipTo.SetJSONData(notification['ADDRESS']);
  xdwdsShipTo.Open;
end;

procedure TFViewAddCustomer.btnAddClick(Sender: TObject);
// Takes the user to the Add Customer Page.
begin
  ShowSelectCustomerForm();
end;


procedure TFViewAddCustomer.btnCancelClick(Sender: TObject);
begin
  ShowConfirmationModal(
    'Are you sure you want to cancel all changes to the customer?',
    'Yes',
    'No',
    procedure(confirmed: Boolean)
    begin
      if confirmed then
      begin
        FViewMain.change := false;
        if CustomerID <> '' then
          FViewMain.ViewAddCustomer(CustomerID, 'Failure: Changes Discarded')
        else
          FViewMain.ShowForm(TFViewCustomers);
      end;
    end);
end;



procedure TFViewAddCustomer.btnClearClick(Sender: TObject);
// Clears the shipping address fields.
begin
  Clear();
end;

procedure TFViewAddCustomer.btnCloseClick(Sender: TObject);
// closes the Add Customer page.
begin
  FViewMain.ShowForm(TFViewCustomers)
end;


procedure TFViewAddCustomer.btnDeleteClick(Sender: TObject);
// Eventually will delete customers after a confirmation
// TODO implement deleting customers
begin
  ShowToast('Deleting Customers Is Not Yet Available', 'info');
  Exit;
end;

procedure TFViewAddCustomer.btnEditClick(Sender: TObject);
// Puts the form into edit mode
begin
  EditMode();
end;

procedure TFViewAddCustomer.edtShippingAddressChange(Sender: TObject);
// Puts the form into Address Edit Mode
begin
  AddressEditMode();
end;


procedure TFViewAddCustomer.wdbtcAddressesDblClickCell(Sender: TObject; ACol,
  ARow: Integer);
// Retrieves the shipping address allowing it to be edited.
begin
  xdwdsShipTo.Locate('ship_id', wdbtcAddresses.Cells[0, ARow], []);

  edtShippingAddress.Text := xdwdsShipTo.FieldByName('shipping_address').AsString;
  edtShippingCity.Text := xdwdsShipTo.FieldByName('city').AsString;
  edtShippingState.Text := xdwdsShipTo.FieldByName('state').AsString;
  edtShippingZip.Text := xdwdsShipTo.FieldByName('zip').AsString;
  edtShippingContact.Text := xdwdsShipTo.FieldByName('contact').AsString;
  memoShipBlock.Text := xdwdsShipTo.FieldByName('ADDRESS').AsString;

  if memoShipBlock.Lines.Count > 0 then
    edtFirstLine.Text := memoShipBlock.Lines[0]
  else
    edtFirstLine.Text := '';

  btnShipEdit.Enabled := true;
  btnShipDelete.Enabled := true;
end;


procedure TFViewAddCustomer.SendCustomerToServer();
// Creates the customer JSON and then sends it to the server.
var
  customerJSON: TJSONObject;
  Field: TField;
  Response: TXDataClientResponse;
  notification: TJSObject;
  input: TJSHTMLElement;
  msg: string;
  BILL_ADDRESS_BLOCK: string;
begin

  customerJSON := TJSONObject.Create;

  if dtpStartDate.Date <> 0 then
    XDataWebDataSet1START_DATE.Value := DateTimeToStr(dtpStartDate.Date);

  if dtpEndDate.Date <> 0 then
    XDataWebDataSet1END_DATE.Value := DateTimeToStr(dtpEndDate.Date);

  XDataWebDataSet1.First;

  while not XDataWebDataSet1.Eof do
  begin
    for Field in XDataWebDataSet1.Fields do
    begin
      if Field is TStringField then
      begin
        if Field.AsString = '' then
          customerJSON.AddPair(Field.FieldName, '')
        else
          customerJSON.AddPair(Field.FieldName, Field.AsString);  // Add all other fields
      end
      else if Field is TIntegerField then
      begin
        customerJSON.AddPair(Field.FieldName, Field.AsString);
      end;
    end;
    XDataWebDataSet1.Next;
  end;

  BILL_ADDRESS_BLOCK := edtName.Text + slinebreak +
                        edtBillContact.Text + slinebreak +
                        edtBillAddress.Text + slinebreak +
                        edtBillCity.Text + ', ' + edtBillState.Text + ' ' + edtBillZip.Text;

  CustomerJSON.AddPair('BILL_ADDRESS_BLOCK', BILL_ADDRESS_BLOCK);

  customerJSON.AddPair('mode', mode);
  if mode = 'EDIT' then
    customerJSON.AddPair('CUSTOMER_ID', customerID);


  Response := await(XDataWebClient1.RawInvokeAsync('ILookupService.AddCustomer',
              [customerJSON.ToString]));

  notification := TJSObject(Response.Result);
  msg := string(notification['status']);
  if CustomerID = '' then
    CustomerID := string(notification['CustomerID']);

  edtCustomerID.Text := CustomerID;

  ShowToast(msg);

  //TODO update the BILL_ADDRESS_BLOCK memo after saving.

  if msg.Contains('Failure') then
  begin
    input := TJSHTMLInputElement(document.getElementById('edtcompanyaccountname'));
    input.classList.add('is-invalid');
    input := TJSHTMLInputElement(document.getElementById('shortnamefeedback'));
    input.innerHTML := 'Company Account Name must be Unique.' ;
  end;

end;

procedure TFViewAddCustomer.btnSaveClick(Sender: TObject);
// Sends the customer JSON to the server
begin
  if VerifyCustomer() then
  begin
    Save();
  end;
end;

procedure TFViewAddCustomer.Save;
begin
  await(sendCustomerToServer());
  await(GetCustomer());
  ViewMode();
end;

procedure TFViewAddCustomer.btnShipAddClick(Sender: TObject);
// Sets the form to address edit mode and allows the user to add a shipping address.
begin
  shipmode := 'ADD';
  Clear();
  AddressEditMode();
end;


procedure TFViewAddCustomer.btnShipCancelClick(Sender: TObject);
begin
  ShowConfirmationModal(
    'Are you sure you want to cancel address editing?',
    'Yes',
    'No',
    procedure(confirmed: Boolean)
    begin
      if confirmed then
      begin
        shipmode := '';
        Clear();
        ViewMode();
      end;
    end);
end;


procedure TFViewAddCustomer.btnShipDeleteClick(Sender: TObject);
begin
  ShowToast('Deleting Shipping Addresses is not yet implemented.', 'info');
  {ShowConfirmationModal(
    'Are you sure you want to delete this address?',
    'Delete',
    'Cancel',
    procedure(confirmed: Boolean)
    begin
      if confirmed then
      begin
        Utils.ShowSpinner('spinner');
        DelAddress();
        clear();
      end;
    end);   }
end;



procedure TFViewAddCustomer.btnShipEditClick(Sender: TObject);
// Sets the form into AddressEdit mode.
begin
  shipmode := 'EDIT';
  AddressEditMode();
end;

[async] procedure TFViewAddCustomer.btnShipSaveClick(Sender: TObject);
begin
  if (customerID = '') then
  begin
    ShowErrorModal('Cannot save address: Customer ID is not set.');
    Exit;
  end;
  console.log('Saving address. CustomerID = ' + customerID);

  if VerifyAddress() then
  begin
    await(SendAddressToServer);
    Clear();
    await(GetCustomer); // Ensures xdwdsShipTo is refreshed with server data
    ViewMode();
  end;
end;


procedure TFViewAddCustomer.GetCustomer;
// Retrieves a customer for a given CustomerID.
var
  xdcResponse: TXDataClientResponse;
  customer, RepUsers : TJSObject;
  items: TJSObject;
  ship_block: TStringList;
begin
  xdcResponse := await(XDataWebClient1.RawInvokeAsync('ILookupService.GetCustomer', [customerID]));
  customer :=  TJSObject(xdcResponse.Result);



  XDataWebDataSet1.Close;
  XDataWebDataSet1.SetJsonData(customer);
  XDataWebDataSet1.Open;


  edtCustomerID.Text := CustomerID;

  if XDataWebDataSet1.FieldByName('START_DATE').AsString <> '' then
    dtpStartDate.Date := XDataWebDataSet1.FieldByName('START_DATE').AsDateTime;

  if XDataWebDataSet1.FieldByName('END_DATE').AsString <> '' then
    dtpEndDate.Date := XDataWebDataSet1.FieldByName('End_DATE').AsDateTime;

  xdwdsShipTo.Close;
  xdwdsShipTo.SetJSONData(customer['SHIPPING_ADDRESS_LIST']);
  xdwdsShipTo.Open;

  memoAddressBlock.Text :=  string(customer['staff_fields_invoice_to']);

  edtShippingAddress.Text := xdwdsShipTo.FieldByName('shipping_address').AsString;
  edtShippingCity.Text := xdwdsShipTo.FieldByName('city').AsString;
  edtShippingState.Text := xdwdsShipTo.FieldByName('state').AsString;
  edtShippingZip.Text := xdwdsShipTo.FieldByName('zip').AsString;
  edtShippingContact.Text := xdwdsShipTo.FieldByName('contact').AsString;
  memoShipBlock.Text := xdwdsShipTo.FieldByName('ADDRESS').AsString;
  if memoShipBlock.Lines.Count > 0 then
    edtFirstLine.Text := memoShipBlock.Lines[0]
  else
    edtFirstLine.Text := '';

  xdcResponse := await(XDataWebClient1.RawInvokeAsync('ILookupService.GetRepUsers', []));
  RepUsers :=  TJSObject(xdcResponse.Result);
  console.log(RepUsers);

  xdwdsUsers.Close;
  xdwdsUsers.SetJSONData(RepUsers['value']);
  xdwdsUsers.Open;
end;

procedure TFViewAddCustomer.tmrReturnTimer(Sender: TObject);
// Timer to returnto the customer page  because it takes slightly too long to
// Delete customers causing ghost customers to show up.
begin
  Utils.HideSpinner('spinner');
  tmrReturn.Enabled := false;
  FViewMain.ViewCustomerList('Success:Customer Successfully Deleted');
end;

function TFViewAddCustomer.VerifyCustomer(): boolean;
var
  input: TJSHTMLElement;
begin
  result := true;

  // Company Verification
  input := TJSHTMLInputElement(document.getElementById('edtcompanyname'));
  if edtName.Text = '' then
  begin
    input.classList.add('is-invalid');
    result := false;
  end
  else
    input.classList.remove('is-invalid');


  input := TJSHTMLInputElement(document.getElementById('edtcompanyaccountname'));
  if edtShortName.Text = '' then
  begin
    input.classList.add('is-invalid');
    result := false;
  end
  else
    input.classList.remove('is-invalid');


  // Billing Information Verification
  input := TJSHTMLInputElement(document.getElementById('edtbillingaddress'));
  if edtBillAddress.Text = '' then
  begin
    input.classList.add('is-invalid');
    result := false;
  end
  else
    input.classList.remove('is-invalid');

  input := TJSHTMLInputElement(document.getElementById('edtbillingcity'));
  if edtBillCity.Text = '' then
  begin
    input.classList.add('is-invalid');
    result := false;
  end
  else
    input.classList.remove('is-invalid');

  input := TJSHTMLInputElement(document.getElementById('edtbillingstate'));
  if edtBillState.Text = '' then
  begin
    input.classList.add('is-invalid');
    result := false;
  end
  else
    input.classList.remove('is-invalid');

  input := TJSHTMLInputElement(document.getElementById('edtbillingzip'));
  if edtBillZip.Text = '' then
  begin
    input.classList.add('is-invalid');
    result := false;
  end
  else
    input.classList.remove('is-invalid');

  input := TJSHTMLInputElement(document.getElementById('edtbillingcontact'));
  if edtBillContact.Text = '' then
  begin
    input.classList.add('is-invalid');
    result := false;
  end
  else
    input.classList.remove('is-invalid');
end;

function TFViewAddCustomer.VerifyAddress: Boolean;
// Verifies all the shipping information is filled in.
var
  input: TJSHTMLElement;
begin
  result := true;
  input := TJSHTMLInputElement(document.getElementById('edtshippingaddress'));
  if edtShippingAddress.Text = '' then
  begin
    input.classList.add('is-invalid');
    result := false;
  end
  else
    input.classList.remove('is-invalid');

  input := TJSHTMLInputElement(document.getElementById('edtshippingcity'));
  if edtShippingCity.Text = '' then
  begin
    input.classList.add('is-invalid');
    result := false;
  end
  else
    input.classList.remove('is-invalid');

  input := TJSHTMLInputElement(document.getElementById('edtshippingstate'));
  if edtShippingState.Text = '' then
  begin
    input.classList.add('is-invalid');
    result := false;
  end
  else
    input.classList.remove('is-invalid');

  input := TJSHTMLInputElement(document.getElementById('edtshippingzip'));
  if edtShippingZip.Text = '' then
  begin
    input.classList.add('is-invalid');
    result := false;
  end
  else
    input.classList.remove('is-invalid');

  input := TJSHTMLInputElement(document.getElementById('edtshippingcontact'));
  if edtShippingContact.Text = '' then
  begin
    input.classList.add('is-invalid');
    result := false;
  end
  else
    input.classList.remove('is-invalid');

  input := TJSHTMLInputElement(document.getElementById('edtfirstline'));
  if edtFirstLine.Text = '' then
  begin
    input.classList.add('is-invalid');
    result := false;
  end
  else
    input.classList.remove('is-invalid');

end;

procedure TFViewAddCustomer.EditMode;
// Enables Customer Fields while disabling shipping address fields.
begin
  XDataWebDataSet1.Edit;
  FViewMain.change := true;
  btnAdd.Enabled := false;
  btnDelete.Enabled := false;
  btnClose.Enabled := false;
  btnSave.Enabled := true;
  btnCancel.Enabled := True;
  btnEdit.Enabled := false;

  // Disable Shipping Address Editting
  edtShippingAddress.Enabled := false;
  edtShippingCity.Enabled := false;
  edtShippingState.Enabled := false;
  edtShippingZip.Enabled := false;
  edtShippingContact.Enabled := false;
  edtFirstLine.Enabled := false;

  btnShipDelete.Enabled := false;
  btnShipSave.Enabled := false;
  btnShipCancel.Enabled := false;
  btnShipEdit.Enabled := false;
  btnShipAdd.Enabled := false;

  lblFormState.Caption := 'Edit Mode';
  lblFormState.ElementHandle.classList.remove('text-danger');
  lblFormState.ElementHandle.classList.add('text-success');
end;

procedure TFViewAddCustomer.AddressEditMode;
// Enables Shipping Address fields while disabling customer fields.
begin
  xdwdsShipTo.Edit;
  FViewMain.change := true;
  btnDelete.Enabled := false;
  btnClose.Enabled := false;
  btnSave.Enabled := false;
  btnCancel.Enabled := false;
  btnEdit.Enabled := false;
  btnAdd.Enabled := false;

  btnShipDelete.Enabled := false;
  btnShipSave.Enabled := true;
  btnShipCancel.Enabled := true;
  btnShipEdit.Enabled := false;
  btnShipAdd.Enabled := false;

  edtShippingAddress.Enabled := true;
  edtShippingCity.Enabled := true;
  edtShippingState.Enabled := true;
  edtShippingZip.Enabled := true;
  edtShippingContact.Enabled := true;
  edtFirstLine.Enabled := true;

  edtName.Enabled := false;
  edtShortName.Enabled := false;
  edtBillAddress.Enabled := false;
  edtBillCity.Enabled := false;
  edtBillState.Enabled := false;
  edtBillZip.Enabled := false;
  edtBillContact.Enabled := false;
  dtpStartDate.Enabled := false;
  dtpEndDate.Enabled := false;
  edtFax.Enabled := false;
  edtPhone.enabled := false;
  wdblcbRep.Enabled := false;

  lblFormState.Caption := 'Edit Address';
  lblFormState.ElementHandle.classList.remove('text-danger');
  lblFormState.ElementHandle.classList.add('text-success');
end;


procedure TFViewAddCustomer.ViewMode;
// Enables Customer Fields while disabling shipping address fields.
begin
  btnAdd.Enabled := true;
  btnDelete.Enabled := true;
  btnClose.Enabled := true;
  btnSave.Enabled := false;
  btnCancel.Enabled := false;
  btnEdit.Enabled := true;
  FViewMain.change := false;

  btnShipAdd.Enabled := true;

  if not xdwdsShipTo.IsEmpty then
  begin
    btnShipDelete.Enabled := true;
    btnShipEdit.Enabled := true;
  end
  else
  begin
    btnShipDelete.Enabled := false;
    btnShipEdit.Enabled := false;
  end;

  btnShipSave.Enabled := false;
  btnShipCancel.Enabled := false;

  edtName.Enabled := true;
  edtShortName.Enabled := true;
  edtBillAddress.Enabled := true;
  edtBillCity.Enabled := true;
  edtBillState.Enabled := true;
  edtBillZip.Enabled := true;
  edtBillContact.Enabled := true;
  dtpStartDate.Enabled := true;
  dtpEndDate.Enabled := true;
  edtFax.Enabled := true;
  edtPhone.Enabled := true;
  wdblcbRep.Enabled := true;

  lblFormState.Caption := 'View Mode';
  lblFormState.ElementHandle.classList.remove('text-success');
  lblFormState.ElementHandle.classList.add('text-danger');
end;

end.