unit Auth.ServiceImpl;

interface

uses
  XData.Service.Common,
  XData.Server.Module,
  Auth.Service,
  Auth.Database,
  Uni, Data.DB, System.Hash, System.IniFiles, System.JSON;

type
  [ServiceImplementation]
  TAuthService = class(TInterfacedObject, IAuthService)
  strict private
    authDB: TAuthDatabase;

    userId: string;
    userName: string;
    userFullName: string;
    userStatus: string;
    userEmail: string;
    userAccessLevel: string;
    userTaskRights: string;
    userPerspectiveId: string;
    userFirstName: string;
    userLastName: string;

    procedure AfterConstruction; override;
    procedure BeforeDestruction; override;

    function CheckUrlLogin(const userId, taskId, urlCode: string): Integer;
    procedure LoadUserFromUrlLoginQuery;
  public
    function Login(const userId, taskId, urlCode: string): string;
    function VerifyVersion(ClientVersion: string): TJSONObject;
  end;

implementation

uses
  System.SysUtils,
  System.DateUtils,
  Bcl.JOSE.Core.Builder,
  Bcl.JOSE.Core.JWT,
  Aurelius.Global.Utils,
  XData.Sys.Exceptions,
  Common.Logging,
  Common.Config;

procedure TAuthService.AfterConstruction;
begin
  inherited;
  try
    authDB := TAuthDatabase.Create(nil);
  except
    on E: Exception do
    begin
      Logger.Log(1, 'Error when creating the Auth database: ' + E.Message);
      raise EXDataHttpException.Create(500, 'Unable to create Auth database: A Server Error has occured!');
    end;
  end;
end;

procedure TAuthService.BeforeDestruction;
begin
  authDB.Free;
  inherited;
end;

function TAuthService.VerifyVersion(ClientVersion: string): TJSONObject;
var
  iniFile: TIniFile;
  webClientVersion: string;
begin
  Result := TJSONObject.Create;
  TXDataOperationContext.Current.Handler.ManagedObjects.Add(Result);

  iniFile := TIniFile.Create(ChangeFileExt(ParamStr(0), '.ini'));
  try
    webClientVersion := iniFile.ReadString('Settings', 'webClientVersion', '');
    Result.AddPair('webClientVersion', webClientVersion);

    if webClientVersion = '' then
    begin
      Result.AddPair('error', 'webClientVersion is not configured.');
      Exit;
    end;

    if ClientVersion <> webClientVersion then
    begin
      Result.AddPair('error',
        'Your browser is running an old version of the app.' + sLineBreak +
        'Please click below to reload.');
    end;
  finally
    iniFile.Free;
  end;
end;

function TAuthService.CheckUrlLogin(const userId, taskId, urlCode: string): Integer;
begin
  Result := 0;

  authDB.uqWebTasksUrl.Close;
  authDB.uqWebTasksUrl.ParamByName('USER_ID').AsString := userId;
  authDB.uqWebTasksUrl.ParamByName('TASK_ID').AsString := taskId;
  authDB.uqWebTasksUrl.ParamByName('URL_CODE').AsString := urlCode;
  authDB.uqWebTasksUrl.Open;

  if authDB.uqWebTasksUrl.IsEmpty then
    Exit;

  if authDB.uqWebTasksUrl.FieldByName('STATUS').AsString <> 'ACTIVE' then
  begin
    Result := 2;
    Exit;
  end;

  LoadUserFromUrlLoginQuery;
  Result := 3;
end;

procedure TAuthService.LoadUserFromUrlLoginQuery;
var
  nameValue: string;
begin
  Self.userId := authDB.uqWebTasksUrl.FieldByName('USER_ID').AsString;
  userName := authDB.uqWebTasksUrl.FieldByName('USER_NAME').AsString;
  userStatus := authDB.uqWebTasksUrl.FieldByName('STATUS').AsString;
  userEmail := authDB.uqWebTasksUrl.FieldByName('EMAIL').AsString;
  userAccessLevel := authDB.uqWebTasksUrl.FieldByName('ACCESS_LEVEL').AsString;
  userTaskRights := authDB.uqWebTasksUrl.FieldByName('TASK_RIGHTS').AsString;
  userPerspectiveId := authDB.uqWebTasksUrl.FieldByName('PERSPECTIVE_ID').AsString;
  userLastName := authDB.uqWebTasksUrl.FieldByName('LAST_NAME').AsString;
  userFirstName := authDB.uqWebTasksUrl.FieldByName('FIRST_NAME').AsString;

  nameValue := Trim(authDB.uqWebTasksUrl.FieldByName('NAME').AsString);
  if nameValue <> '' then
    userFullName := nameValue
  else
    userFullName := Trim(userFirstName + ' ' + userLastName);
end;

function TAuthService.Login(const userId, taskId, urlCode: string): string;
var
  userState: Integer;
  jwt: TJWT;
begin
  Logger.Log(3, Format('AuthService.Login - UserID: "%s", TaskID: "%s"', [userId, taskId]));

  try
    userState := CheckUrlLogin(userId, taskId, urlCode);
  except
    on E: Exception do
    begin
      Logger.Log(1, 'URL Login failed due to database error: ' + E.Message);
      raise EXDataHttpException.Create(500, 'Login failed: Unable to connect to the database.');
    end;
  end;

  if userState = 0 then
  begin
    Logger.Log(2, 'Login Error: Invalid code or expired link');
    raise EXDataHttpUnauthorized.Create('Invalid code or expired link');
  end;

  if userState = 2 then
  begin
    Logger.Log(2, 'Login Error: User not active!');
    raise EXDataHttpUnauthorized.Create('User not active!');
  end;

  jwt := TJWT.Create;
  try
    jwt.Claims.JWTId := LowerCase(Copy(TUtils.GuidToVariant(TUtils.NewGuid), 2, 36));
    jwt.Claims.IssuedAt := Now;
    jwt.Claims.Expiration := IncHour(Now, 24);

    jwt.Claims.SetClaimOfType<string>('user_id', Self.userId);
    jwt.Claims.SetClaimOfType<string>('user_name', userName);
    jwt.Claims.SetClaimOfType<string>('user_fullname', userFullName);
    jwt.Claims.SetClaimOfType<string>('user_status', userStatus);
    jwt.Claims.SetClaimOfType<string>('user_email', userEmail);
    jwt.Claims.SetClaimOfType<string>('user_access_level', userAccessLevel);
    jwt.Claims.SetClaimOfType<string>('task_rights', userTaskRights);
    jwt.Claims.SetClaimOfType<string>('user_perspective_id', userPerspectiveId);
    jwt.Claims.SetClaimOfType<string>('first_name', userFirstName);
    jwt.Claims.SetClaimOfType<string>('last_name', userLastName);

    Result := TJOSE.SHA256CompactToken(serverConfig.jwtTokenSecret, jwt);
  finally
    jwt.Free;
  end;
end;

initialization
  RegisterServiceType(TAuthService);

end.
