program templateXDataServer;

uses
  Vcl.Forms,
  FastMM4,
  System.SyncObjs,
  System.SysUtils,
  Vcl.StdCtrls,
  IniFiles,
  Main in 'source\Main.pas' {FMain},
  Common.Config in 'source\Common.Config.pas',
  Common.Ini in 'source\Common.Ini.pas',
  Common.Logging in 'source\Common.Logging.pas',
  Common.Middleware.Logging in 'source\Common.Middleware.Logging.pas',
  Api.Server.Module in 'source\Api.Server.Module.pas' {ApiServerModule: TDataModule},
  Api.Database in 'source\Api.Database.pas' {ApiDatabase: TDataModule},
  Api.Service in 'source\Api.Service.pas',
  Api.ServiceImpl in 'source\Api.ServiceImpl.pas',
  Auth.Database in 'source\Auth.Database.pas' {AuthDatabase: TDataModule},
  Auth.Service in 'source\Auth.Service.pas',
  Auth.ServiceImpl in 'source\Auth.ServiceImpl.pas',
  uLibrary in 'source\uLibrary.pas',
  App.Server.Module in 'source\App.Server.Module.pas' {AppServerModule: TDataModule},
  Auth.Server.Module in 'source\Auth.Server.Module.pas' {AuthServerModule: TDataModule};

type
  TMemoLogAppender = class(TInterfacedObject, ILogAppender)
  private
    FLogLevel: Integer;
    FLogMemo: TMemo;
    FCriticalSection: TCriticalSection;
  public
    constructor Create(ALogLevel: Integer; ALogMemo: TMemo);
    destructor Destroy; override;
    procedure Send(logLevel: Integer; Log: ILog);
  end;

  TFileLogAppender = class(TInterfacedObject, ILogAppender)
  private
    FLogLevel: Integer;
    FFilename: string;
    FLogDirectory: string;
    FCriticalSection: TCriticalSection;
    function GetLogFilePath: string;
  public
    constructor Create(ALogLevel: Integer; AFilename: string);
    destructor Destroy; override;
    procedure Send(logLevel: Integer; Log: ILog);
  end;

{ TMemoLogAppender }

constructor TMemoLogAppender.Create(ALogLevel: Integer; ALogMemo: TMemo);
begin
  FLogLevel := ALogLevel;
  FLogMemo := ALogMemo;
  FCriticalSection := TCriticalSection.Create;
end;

destructor TMemoLogAppender.Destroy;
begin
  FCriticalSection.Free;
  inherited;
end;

procedure TMemoLogAppender.Send(logLevel: Integer; Log: ILog);
var
  logMsg: string;
  logTime: TDateTime;
  formattedMessage: string;
begin
  FCriticalSection.Acquire;
  try
    logTime := Now;

    formattedMessage := FormatDateTime('[yyyy-mm-dd HH:nn:ss.zzz]', logTime);
    logMsg := Log.GetMessage;
    if logMsg.IsEmpty then
      formattedMessage := ''
    else
      formattedMessage := formattedMessage + '[' + IntToStr(logLevel) + '] ' + logMsg;

    if logLevel <= FLogLevel then
      FLogMemo.Lines.Add(formattedMessage);
  finally
    FCriticalSection.Release;
  end;
end;

{ TFileLogAppender }

constructor TFileLogAppender.Create(ALogLevel: integer; AFilename: string);
var
  iniFile: TIniFile;
  fileNum: integer;
  logsDir: string;
begin
  FLogLevel := ALogLevel;
  FCriticalSection := TCriticalSection.Create;

  logsDir := ExtractFilePath(Application.ExeName) + 'logs\';

  if logsDir = '' then
    raise Exception.Create('logsDir is blank. ExeName="' + Application.ExeName + '"');

  if not DirectoryExists(logsDir) then
    CreateDir(logsDir);

  iniFile := TIniFile.Create(ChangeFileExt(Application.ExeName, '.ini'));
  try
    fileNum := iniFile.ReadInteger('Settings', 'LogFileNum', 0);
    FFilename := AFilename + Format('%.4d', [fileNum]);
    iniFile.WriteInteger('Settings', 'LogFileNum', fileNum + 1);
  finally
    iniFile.Free;
  end;

  FLogDirectory := logsDir;
end;

destructor TFileLogAppender.Destroy;
begin
  FCriticalSection.Free;
  inherited;
end;

function TFileLogAppender.GetLogFilePath: string;
begin
  Result := FLogDirectory + FFilename + '.log';
end;

procedure TFileLogAppender.Send(logLevel: Integer; Log: ILog);
var
  logFile: TextFile;
  logMsg: string;
  logTime: TDateTime;
  formattedMessage: string;
begin
  if logLevel > FLogLevel then
    Exit;

  FCriticalSection.Acquire;
  try
    logTime := Now;

    formattedMessage := FormatDateTime('[yyyy-mm-dd HH:nn:ss.zzz]', logTime);
    logMsg := Log.GetMessage;
    if logMsg.IsEmpty then
      formattedMessage := ''
    else
      formattedMessage := formattedMessage + '[' + IntToStr(logLevel) + '] ' + logMsg;

    AssignFile(logFile, GetLogFilePath);
    if FileExists(GetLogFilePath) then
      Append(logFile)
    else
      Rewrite(logFile);

    try
      Writeln(logFile, formattedMessage);
    finally
      CloseFile(logFile);
    end;
  finally
    FCriticalSection.Release;
  end;
end;

{$R *.res}

begin
  ReportMemoryLeaksOnShutdown := True;
  Application.Initialize;
  Application.MainFormOnTaskbar := True;
  Application.CreateForm(TFMain, FMain);
  LoadIniEntries;
  Logger.AddAppender(TMemoLogAppender.Create(IniEntries.consoleLogLevel, FMain.memoLogs));
  Logger.AddAppender(TFileLogAppender.Create(IniEntries.fileLogLevel, 'templateServer'));

  Application.Run;

end.

