// Implementation of the Lookup Service interface used to send call information
// to the client.
// Authors:
// Cameron Hayes
// Mac ...
// Elias Sarraf
unit Lookup.ServiceImpl;

interface

uses
  XData.Server.Module, XData.Service.Common, Api.Database, Data.DB,
  Lookup.Service, System.Hash, System.Classes, Common.Logging;


type
  [ServiceImplementation]
  TLookupService = class(TInterfacedObject, ILookupService)
  strict private
  callsDB: TApiDatabaseModule;
  private
    function GetUsers(searchOptions: string): TUserList;
    function GetCalls(searchOptions: string): TCallList;
    function EditUser(const editOptions: string): string;
    function Search(phoneNum: string): TCallList;
    function AddUser(userInfo: string): string;
    function DelUser(username: string): string;
    procedure AfterConstruction; override;
    procedure BeforeDestruction; override;
  end;

implementation

uses
  System.SysUtils,
  System.Generics.Collections,
  XData.Sys.Exceptions, uLibrary;

procedure TLookupService.AfterConstruction;
begin
  inherited;
  callsDB := TApiDatabaseModule.Create(nil);
  Logger.Log(3, 'callsDB created');
end;

procedure TLookupService.BeforeDestruction;
begin
  callsDB.Free;
  inherited;
  Logger.Log(3, 'callsDB destroyed');
end;

function TLookupService.Search(phoneNum: string): TCallList;
var
  SQL: string;
  call: TCallItem;
begin
  Logger.Log(3, 'Search called for phone number: ' + phoneNum);

  SQL := 'select * ' +
         'from calls inner join recordings on calls.sid = recordings.call_sid ' +
         'where from_formatted = ' + QuotedStr(phoneNum) +
         ' order by calls.date_created desc';
  Logger.Log(3, 'Search main query: ' + SQL);

  doQuery(callsDB.UniQuery1, SQL);

  Result := TCallList.Create;
  Result.data := TList<TCallItem>.Create;
  TXDataOperationContext.Current.Handler.ManagedObjects.Add(Result.data);

  while not callsDB.UniQuery1.Eof do
  begin
    call := TCallItem.Create;
    TXDataOperationContext.Current.Handler.ManagedObjects.Add(call);
    Result.data.Add(call);
    call.callSid := callsDB.UniQuery1.FieldByName('sid').AsString;
    call.fromNumber := callsDB.UniQuery1.FieldByName('from_formatted').AsString;
    call.ToNumber := callsDB.UniQuery1.FieldByName('to_formatted').AsString;
    call.dateCreated := callsDB.UniQuery1.FieldByName('date_created').AsString;
    call.mediaUrl := callsDB.UniQuery1.FieldByName('media_url').AsString;
    call.duration := callsDB.UniQuery1.FieldByName('duration').AsString;
    call.transcription := callsDB.UniQuery1.FieldByName('transcription').AsString;
    callsDB.UniQuery1.Next;
  end;

  callsDB.UniQuery1.Close;

  SQL := 'select count(*) as total_count from calls inner join ' +
         'recordings on calls.sid = recordings.call_sid where ' +
         'from_formatted = ' + QuotedStr(phoneNum);
  Logger.Log(3, 'Search count query: ' + SQL);

  doQuery(callsDB.UniQuery1, SQL);
  Result.count := callsDB.UniQuery1.FieldByName('total_count').AsInteger;
  callsDB.UniQuery1.Close;

  Logger.Log(3, 'Search completed. Total results: ' + IntToStr(Result.count));
end;



function TLookupService.GetCalls(searchOptions: string): TCallList;
var
  params: TStringList;
  SQL: string;
  DBString: string;
  call: TCallItem;
  offset: string;
  limit: string;
  PhoneNum: string;
  PageNum: integer;
  PageSize: integer;
  StartDate: string;
  EndDate: string;
  OrderBy: string;
  whereSQL: string;
  orderBySQL: string;
  Caller: string;
begin
  Logger.Log(3, 'GetCalls called with searchOptions: ' + searchOptions);

  params := TStringList.Create;
  params.StrictDelimiter := true;
  params.Delimiter := '&';
  params.DelimitedText := searchOptions;

  PhoneNum := params.Values['phonenumber'];
  PageNum := StrToInt(params.Values['pagenumber']);
  PageSize := StrToInt(params.Values['pagesize']);
  StartDate := params.Values['startdate'];
  EndDate := params.Values['enddate'];
  OrderBy := params.Values['orderby'];
  Caller := params.Values['caller'];

  offset := IntToStr((PageNum - 1) * PageSize);
  limit := IntToStr(PageSize);

  whereSQL := 'where ';

  if PhoneNum <> '' then
    whereSQL := whereSQL + 'to_formatted = ' + QuotedStr(PhoneNum);

  if StartDate <> '' then
    if whereSQL = 'where ' then
      whereSQL := whereSQL + 'calls.date_created > ' + QuotedStr(StartDate)
    else
      whereSQL := whereSQL + ' AND calls.date_created > ' + QuotedStr(StartDate);

  if EndDate <> '' then
    if whereSQL = 'where ' then
      whereSQL := whereSQL + 'calls.date_created < ' + QuotedStr(EndDate)
    else
      whereSQL := whereSQL + ' AND calls.date_created < ' + QuotedStr(EndDate);

  if Caller <> '' then
    if whereSQL = 'where ' then
      whereSQL := whereSQL + 'from_formatted = ' + QuotedStr(Caller)
    else
      whereSQL := whereSQL + ' AND from_formatted = ' + QuotedStr(Caller);

  if whereSQL = 'where ' then
    whereSQL := '';

  if (OrderBy = '') or (OrderBy = 'Date') then
    orderBySQL := 'order by calls.date_created desc'
  else
    orderBySQL := 'order by calls.from_formatted desc';

  SQL := 'select * ' +
         'from calls inner join recordings on calls.sid = recordings.call_sid ' +
         whereSQL + ' ' + orderBySQL + ' limit ' + limit + ' offset ' + offset;

  Logger.Log(3, 'GetCalls query: ' + SQL);

  doQuery(callsDB.UniQuery1, SQL);

  Result := TCallList.Create;
  Result.data := TList<TCallItem>.Create;
  TXDataOperationContext.Current.Handler.ManagedObjects.Add(Result.data);

  while not callsDB.UniQuery1.Eof do
  begin
    call := TCallItem.Create;
    TXDataOperationContext.Current.Handler.ManagedObjects.Add(call);
    Result.data.Add(call);
    call.callSid := callsDB.UniQuery1.FieldByName('sid').AsString;
    call.fromNumber := callsDB.UniQuery1.FieldByName('from_formatted').AsString;
    call.ToNumber := callsDB.UniQuery1.FieldByName('to_formatted').AsString;
    call.dateCreated := callsDB.UniQuery1.FieldByName('date_created').AsString;
    call.mediaUrl := callsDB.UniQuery1.FieldByName('media_url').AsString;
    call.duration := callsDB.UniQuery1.FieldByName('duration').AsString;
    call.transcription := callsDB.UniQuery1.FieldByName('transcription').AsString;
    callsDB.UniQuery1.Next;
  end;

  callsDB.UniQuery1.Close;

  SQL := 'select count(*) as total_count from calls inner join recordings on calls.sid = recordings.call_sid ' + whereSQL;
  Logger.Log(3, 'GetCalls count query: ' + SQL);

  doQuery(callsDB.UniQuery1, SQL);
  Result.count := callsDB.UniQuery1.FieldByName('total_count').AsInteger;
  callsDB.UniQuery1.Close;

  Logger.Log(3, 'GetCalls completed successfully. Returned count: ' + IntToStr(Result.count));
end;


function TLookupService.GetUsers(searchOptions: string): TUserList;
var
  SQL: string;
  user: TUserItem;
begin
  if searchOptions = '' then
    SQL := 'select * from users order by full_name ASC'
  else
    SQL := 'select * from users where username=' + QuotedStr(searchOptions);

  Logger.Log(3, 'GetUsers query: ' + SQL);

  doQuery(callsDB.UniQuery1, SQL);

  Result := TUserList.Create;
  Result.data := TList<TUserItem>.Create;
  TXDataOperationContext.Current.Handler.ManagedObjects.Add(Result.data);

  while not callsDB.UniQuery1.Eof do
  begin
    user := TUserItem.Create;
    TXDataOperationContext.Current.Handler.ManagedObjects.Add(user);
    Result.data.Add(user);
    user.userID := callsDB.UniQuery1.FieldByName('user_id').AsString;
    user.username := callsDB.UniQuery1.FieldByName('username').AsString;
    user.full_name := callsDB.UniQuery1.FieldByName('full_name').AsString;
    user.phone_number := callsDB.UniQuery1.FieldByName('phone_number').AsString;
    user.email_address := callsDB.UniQuery1.FieldByName('email').AsString;
    user.admin := callsDB.UniQuery1.FieldByName('admin').AsBoolean;
    user.password := callsDB.UniQuery1.FieldByName('password').AsString;
    user.active := callsDB.UniQuery1.FieldByName('active').AsBoolean;
    callsDB.UniQuery1.Next;
  end;

  callsDB.UniQuery1.Close;

  SQL := 'select count(*) as total_count from users';
  Logger.Log(3, 'GetUsers count query: ' + SQL);
  doQuery(callsDB.UniQuery1, SQL);
  Result.count := callsDB.UniQuery1.FieldByName('total_count').AsInteger;
  callsDB.UniQuery1.Close;

  Logger.Log(3, 'GetUsers returned user list, count = ' + IntToStr(Result.count));
end;


function TLookupService.EditUser(const editOptions: string): string;
var
  params: TStringList;
  user, full_name, email, phone, Admin, newUser, hashString, hashPW, password, active: string;
  SQL: string;
begin
  Logger.Log(3, 'EditUser called with options: ' + editOptions);
  params := TStringList.Create;
  params.Delimiter := '&';
  params.StrictDelimiter := true;
  params.DelimitedText := editOptions;

  user := params.Values['username'];
  full_name := params.Values['fullname'];
  phone := params.Values['phonenumber'];
  email := params.Values['email'];
  Admin := params.Values['admin'];
  newUser := params.Values['newuser'];
  password := params.Values['password'];
  active := params.Values['active'];

  SQL := 'select * from users where username = ' + QuotedStr(user);
  Logger.Log(3, 'EditUser query: ' + SQL);
  doQuery(callsDB.UniQuery1, SQL);

  if callsDB.UniQuery1.IsEmpty then
  begin
    Result := 'No such user found';
    Logger.Log(2, 'EditUser failed: user not found (' + user + ')');
  end
  else
  begin
    callsDB.UniQuery1.Edit;

    if not newUser.IsEmpty then
      callsDB.UniQuery1.FieldByName('username').AsString := newUser;

    if not full_name.IsEmpty then
      callsDB.UniQuery1.FieldByName('full_name').AsString := full_name;

    if not phone.IsEmpty then
      callsDB.UniQuery1.FieldByName('phone_number').AsString := phone;

    if not email.IsEmpty then
      callsDB.UniQuery1.FieldByName('email').AsString := email;

    if not Admin.IsEmpty then
      callsDB.UniQuery1.FieldByName('admin').AsBoolean := StrToBool(Admin);

    if not Active.IsEmpty then
      callsDB.UniQuery1.FieldByName('active').AsBoolean := StrToBool(Active);

    if (password <> 'hidden') and (not password.IsEmpty) then
    begin
      hashString := callsDB.UniQuery1.FieldByName('date_created').AsString + password;
      hashPW := THashSHA2.GetHashString(hashString, THashSHA2.TSHA2Version.SHA512).ToUpper;
      callsDB.UniQuery1.FieldByName('password').AsString := hashPW;
    end;

    callsDB.UniQuery1.Post;
    Result := 'Success:Edit Successful';
    Logger.Log(3, 'EditUser success for: ' + user);
  end;

  callsDB.UniQuery1.Close;
end;



function TLookupService.AddUser(userInfo: string): string;
var
  dateCreated: TDateTime;
  hashString, hashPW, SQL: string;
  params: TStringList;
begin
  Logger.Log(3, 'AddUser called with info: ' + userInfo);

  params := TStringList.Create;
  try
    params.StrictDelimiter := true;
    params.Delimiter := '&';
    params.DelimitedText := userInfo;

    dateCreated := Now;
    hashString := DateTimeToStr(dateCreated) + params.Values['password'];
    hashPW := THashSHA2.GetHashString(hashString, THashSHA2.TSHA2Version.SHA512).ToUpper;

    SQL := 'select * from users where username = ' + QuotedStr(params.Values['username'].ToLower);
    Logger.Log(3, 'AddUser query: ' + SQL);

    callsDB.UniQuery1.Close;
    callsDB.UniQuery1.SQL.Text := SQL;
    callsDB.UniQuery1.Open;

    if callsDB.UniQuery1.IsEmpty then
    begin
      callsDB.UniQuery1.Insert;
      callsDB.UniQuery1.FieldByName('username').AsString := params.Values['username'].ToLower;
      callsDB.UniQuery1.FieldByName('password').AsString := hashPW;
      callsDB.UniQuery1.FieldByName('date_created').AsString := DateTimeToStr(dateCreated);
      callsDB.UniQuery1.FieldByName('full_name').AsString := params.Values['fullname'];
      callsDB.UniQuery1.FieldByName('phone_number').AsString := params.Values['phonenumber'];
      callsDB.UniQuery1.FieldByName('email').AsString := params.Values['email'];
      callsDB.UniQuery1.FieldByName('admin').AsBoolean := StrToBool(params.Values['admin']);
      callsDB.UniQuery1.FieldByName('active').AsBoolean := True;
      callsDB.UniQuery1.Post;

      Result := 'Success:User successfully added';
      Logger.Log(3, 'AddUser success: ' + params.Values['username']);
    end
    else
    begin
      Result := 'Failure:Username already taken';
      Logger.Log(2, 'AddUser failed: Username already taken (' + params.Values['username'] + ')');
    end;

  finally
    params.Free;
  end;
end;



function TLookupService.DelUser(username: string): string;
var
  SQL: string;
begin
  Logger.Log(3, 'DelUser called for: ' + username);
  SQL := 'select * from users where username = ' + QuotedStr(username.ToLower);
  callsDB.UniQuery1.Close;
  callsDB.UniQuery1.SQL.Text := SQL;
  callsDB.UniQuery1.Open;

  if callsDB.UniQuery1.IsEmpty then
  begin
    Result := 'Failure:User does not exist';
    Logger.Log(2, 'DelUser failed: user not found (' + username + ')');
  end
  else
  begin
    SQL := 'DELETE FROM users where username = ' + QuotedStr(username.ToLower);
    callsDB.UniQuery1.SQL.Text := SQL;
    callsDB.UniQuery1.ExecSQL;
    Result := 'Success:User deleted';
    Logger.Log(3, 'DelUser success: ' + username);
  end;
end;


initialization
  RegisterServiceType(TLookupService);

end.

