unit uLibrary;

interface

uses
  System.Classes, Uni;

const
  ADD_REC_AUDIT_ENTRY = '0';
  EDIT_REC_AUDIT_ENTRY = '1';
  DEL_REC_AUDIT_ENTRY = '2';
  REVIEW_REC_AUDIT_ENTRY = '3';
  VIEW_REC_AUDIT_ENTRY = '4';
  FIND_REC_AUDIT_ENTRY = '5';
  PRINT_REC_AUDIT_ENTRY = '6';
  OTHER_REC_AUDIT_ENTRY = '99';

function GetServerTimeStamp( uq: TUniQuery ): TDateTime;
procedure DoQuery( uq: TUniQuery; sql: string );
function CalculateAge( const dob, dt: TDateTime ): Integer;
function GetNextSeqVal( uq: TUniQuery; sequence: string ): string;
function FormatNamePersonnel( uq: TUniQuery; format: string ): string;
function FormatBkNum( bkNum: string ): string;
function GetAssociatedNumber( uq: TUniQuery; numberType: string ): string;
function FormatBookingAddress( uq: TUniQuery; format: string ): string;
function SetMasterAuditEntry( uq: TUniQuery; const entryId, auditType, linkId, agency, personnelId, recUser, details, searchKey, execSource: string ): Boolean;
function SetDetailAuditEntry( uq: TUniQuery; const entryId, title, auditType: string; auditList: TStringList ): Boolean;
function GetOfficerName( agency, officer: string; uq: TUniQuery ): string;
function GetRiciOfficerName( agency, officer: string; uq: TUniQuery ): string;

implementation

uses
  System.SysUtils,
  Data.DB;

function GetServerTimeStamp( uq: TUniQuery ): TDateTime;
var
  sql: string;
  serverDateTime: TDateTime;
begin
  sql := 'select sysdate as currentdatetime from dual';

  DoQuery( uq, sql );
  serverDateTime := uq.FieldByName('CURRENTDATETIME').AsDateTime;
  uq.Close;
  Result := serverDateTime;
end;

procedure DoQuery(uq: TUniQuery; sql: string);
begin
  uq.Close;
  uq.SQL.Text := sql;
  uq.Open;
end;

function CalculateAge( const dob, dt: TDateTime): Integer;
var
  age: Integer;
  y1, m1, d1, y2, m2, d2: Word;
begin
  Result := 0;

  if dt < dob then
    Exit;

  DecodeDate( dob, y1, m1, d1);
  DecodeDate( dt, y2, m2, d2);
  age := y2 - y1;
  // Feb 29
  //if ( (m1=2) and (d1=29) ) and ( not IsLeapYear(y2) ) then
  //  d1 := 28;

  if (m1 = 2) and (d1 = 29) and (not (IsLeapYear (y2))) then
  begin
    m1 := 3;
    d1 := 1;
  end;

  if (m2 < m1) or ((m2 = m1) and (d2 < d1)) then
    Dec(age);
  Result := age
end;

function GetNextSeqVal(uq: TUniQuery; sequence: string ): string;
var
  sql: string;
begin
  sql := 'select ' + sequence + '.NEXTVAL as nextseqval from dual';
  uq.Close;
  uq.SQL.Text :=  sql;
  uq.Open;
  Result := uq.FieldByName('NEXTSEQVAL').AsString;
end;

function FormatNamePersonnel( uq: TUniQuery; format: string ): string;
var
  leng: Integer;
  i: Integer;
  officerText: String;
begin
  leng := Length( format );

  for i := 0 to leng - 1 do
  begin
    case format[i+1] of
      'S':
          officerText := officerText + uq.FieldByName('PF_LNAME').AsString;
      'F':
        if not uq.FieldByName('PF_FNAME').AsString.IsEmpty then
          officerText := TrimRight( officerText + uq.FieldByName('PF_FNAME').AsString ) ;
      'M':
        if not uq.FieldByName('PF_MI').AsString.IsEmpty then
          officerText := TrimRight( officerText + uq.FieldByName('PF_MI').AsString );
      ',':
        officerText := officerText + ',';
      '.':
        officerText := officerText +  '.';
      ' ':
        officerText := officerText +  ' ';
    end;
  end;

  Result := officerText;
end;

function FormatBkNum( bkNum: string ): string;
var
  bkNumStr: string;
begin
  bkNumStr := bkNum;
  Result := bkNumStr.Insert( 4, '-' );
end;

function GetAssociatedNumber( uq: TUniQuery; numberType: string): string;
var
  TLocateOptions: set of TLocateOption;
begin
  if uq.Locate('OTHER_AGENCY_CODE', numberType, TLocateOptions)
    then Result := uq.FieldByName('IDENTIFICATION').AsString
end;

function FormatBookingAddress( uq: TUniQuery; format: string ): string;
var
  addressText: AnsiString;
  leng: Integer;
  i : Integer;
begin
  leng := Length( format );

  for i := 0 to leng - 1 do
  begin
    case format[i+1] of
      'S':
        begin
          addressText := addressText + uq.FieldByName('STREET_NUM').AsString;
          if uq.FieldByName('STREET_NUM_HALF').AsString = 'Y' then
            addressText := addressText + ' 1/2';
          if uq.FieldByName('STREET_DIRECTION').AsString <> '' then
            addressText := addressText + ' ' + uq.FieldByName('STREET_DIRECTION').AsString;
          if uq.FieldByName('STREET_NAME').AsString <> '' then
            addressText := addressText + ' ' + TrimRight( uq.FieldByName('STREET_NAME').AsString );
          if uq.FieldByName('STREET_TYPE').AsString <> '' then
            addressText := addressText + ' ' + TrimRight( uq.FieldByName('STREET_TYPE').AsString );
          if uq.FieldByName('APARTMENT_NUM').AsString <> '' then
            addressText := addressText + ' APT: ' + TrimRight( uq.FieldByName('APARTMENT_NUM').AsString );
        end;
      'C':
        if uq.FieldByName('CITY').AsString <> '' then
          addressText := addressText + ' ' + TrimRight( uq.FieldByName('CITY').AsString );
      'T':
        if uq.FieldByName('STATE').AsString <> '' then
          addressText := addressText + ' ' + TrimRight( uq.FieldByName('STATE').AsString );
      'Z':
        if uq.FieldByName('ZIP_CODE').AsString <> '' then
          addressText := addressText + '  ' + TrimRight( uq.FieldByName('ZIP_CODE').AsString );
      'R':
        if uq.FieldByName('COUNTRY').AsString <> '' then
          addressText := addressText + ' ' + TrimRight( uq.FieldByName('COUNTRY').AsString );
      ',':
        addressText := addressText + ',';
      '.':
        addressText := addressText + '.';
      ' ':
        addressText := addressText + ' ';
    end;
  end;

  Result := addressText;
end;

function SetMasterAuditEntry(uq: TUniQuery; const entryId, auditType, linkId, agency, personnelId, recUser, details, searchKey, execSource: string) : Boolean;
var
  sql: string;
begin
  sql := 'insert into auditmaster ';
  sql := sql + '( AUDITMASTERID, SOURCEID, AUDITTYPE, AGENCY, PERSONNELID, RECUSER, RECDATE, DETAILS, SEARCHKEY, EXECSRC) ';
  sql := sql + 'values (';
  sql := sql + entryID + ', ';
  sql := sql + QuotedStr(linkID) + ', ';
  sql := sql + QuotedStr(auditType) + ', ';
  sql := sql + QuotedStr(agency) + ', ';
  sql := sql + personnelid + ', ';
  sql := sql + QuotedStr(recUser) + ', ';
  sql := sql + 'sysdate, ';
  sql := sql + QuotedStr(details) + ', ';
  sql := sql + QuotedStr(searchKey) + ', ';
  sql := sql + QuotedStr(execSource) + ')';

  uq.Close;
  uq.SQL.Text := sql;
  uq.Execute;
  uq.Close;

  Result := True;
end;

function SetDetailAuditEntry(uq: TUniQuery; const entryId, title, auditType: string; auditList: TStringList) : Boolean;
var
  i: Integer;
  sql: string;
begin
  for i := 0 to auditList.Count - 1 do
  begin
    sql := 'insert into auditdetail values (';
    sql := sql + entryId + ', ';
    sql := sql + QuotedStr( auditList.Names[i] ) + ', ';
    sql := sql + QuotedStr( '' ) + ', ';
    sql := sql + QuotedStr( auditList.ValueFromIndex[i] ) + ', ';
    sql := sql + auditType + ')';

    uq.Close;
    uq.SQL.Text := sql;
    uq.Execute;
    uq.Close;
  end;

  Result := True;
end;

function GetOfficerName( agency, officer: string; uq: TUniQuery ): string;
var
  sql: string;
begin
  if agency.IsEmpty or officer.IsEmpty then
    Exit;

  sql :=  'select a.agency_id, p.agency, p.pf_nameid, pf_lname, pf_fname, pf_mi, pf_badge ';
  sql := sql + 'from personnel p ';
  sql := sql + 'join agencycodes a on a.agency = p.agency ';
  sql := sql + 'where a.agency_id = ' + agency + ' and p.pf_nameid = ' + officer;

  uq.Close;
  uq.SQL.Text := sql;
  uq.Open;

  if uq.IsEmpty then
    Result := agency + '-' + officer + ': not found'
  else
  begin
    Result := uq.FieldByName('pf_lname').AsString + ', ' + uq.FieldByName('pf_fname').AsString;
    Result := Result + ' ' + uq.FieldByName('pf_mi').AsString + ' (' + uq.FieldByName('pf_badge').AsString + ')';
  end;
end;

function GetRiciOfficerName( agency, officer: string; uq: TUniQuery ): string;
var
  sql: string;
begin
  if agency.IsEmpty or officer.IsEmpty then
    Exit;

  sql :=  'select * from rici.officer@rici_link where agency = ' + agency + ' and empno = ' + QuotedStr(officer);

  uq.Close;
  uq.SQL.Text := sql;
  uq.Open;

  if uq.IsEmpty then
    Result := agency + '-' + officer + ': not found'
  else
    Result := uq.FieldByName('surname').AsString + ', ' + uq.FieldByName('given1').AsString + ' (' + uq.FieldByName('empno').AsString + ')';
end;

end.
