unit View.Complaints;

interface

uses
  System.SysUtils, System.Classes, Web, WEBLib.Graphics, WEBLib.Forms, WEBLib.Dialogs,
  Vcl.Controls, Vcl.StdCtrls, WEBLib.StdCtrls, WEBLib.Controls, WEBLib.Grids, WebLib.Lists,
  XData.Web.Client, WEBLib.ExtCtrls, DB, XData.Web.JsonDataset,
  XData.Web.Dataset, XData.Web.Connection, Vcl.Forms, DateUtils, WebAudio;

type
  TFViewComplaints = class(TWebForm)
    XDataWebClient1: TXDataWebClient;
    XDataWebDataSet1: TXDataWebDataSet;
    lblEntries: TWebLabel;
    wcbPageSize: TWebComboBox;
    wcbLocation: TWebLookupComboBox;
    dtpStartDate: TWebEdit;
    dtpEndDate: TWebEdit;
    wcbSortBy: TWebComboBox;
    btnApply: TWebButton;
    edtSearch: TWebEdit;
    procedure WebFormCreate(Sender: TObject);
    procedure btnApplyClick(Sender: TObject);
    procedure btnSearchClick(Sender: TObject);
  private
    FChildForm: TWebForm;
    procedure AddRowToTable(PhoneNumber, Caller, Time, Duration, Transcript, MediaUrl: string);
    procedure ClearTable();
    procedure GeneratePagination(TotalPages: Integer);
    function GenerateSearchOptions(): string;
    [async] procedure Search(searchOptions: string);
    [async] procedure GetCalls(searchOptions: string);
    var
      PageNumber: integer;
      PageSize: integer;
      TotalPages: integer;
      StartDate: string;
      EndDate: string;
      OrderBy: string;
      Caller: string;
  public

  end;

var
  FViewComplaints: TFViewComplaints;

implementation

uses
  JS, XData.Model.Classes,
  ConnectionModule, Utils;

{$R *.dfm}

procedure TFViewComplaints.WebFormCreate(Sender: TObject);
// Initializes important values:
// PageNumber: What page number the user is on IE 1: 1-10, 2: 11-20 etc
// TotalPages: Total number of pages returned from the search.
// PageSize: Number of entries per page.
begin
  DMConnection.ApiConnection.Connected := True;
  PageNumber := 1;
  TotalPages := 1; // Initial total pages
  wcbPageSize.Text := '10';
  wcbLocation.DisplayText := 'All';
  wcbSortBy.Text := 'Date';


  // hardcoded default toNumber/location and load grid
  wcbLocation.Value := 'Galleria';
  PageSize := StrToInt(wcbPageSize.Text);
  StartDate := '';
  EndDate   := '';
  OrderBy   := wcbSortBy.Text;
  Caller    := '';
  GetCalls(GenerateSearchOptions);
end;

procedure TFViewComplaints.AddRowToTable(PhoneNumber, Caller, Time, Duration, Transcript, MediaUrl: string);
// Adds rows to the table
// PhoneNumber: phone number of the location
// Caller: phone number of the caller
// Duration: duration of the call
// Transcript: transcription of the recording
// MediaUrl: Link to the recording
var
  NewRow, Cell, P, Button, Audio: TJSHTMLElement;
begin
  NewRow := TJSHTMLElement(document.createElement('tr'));

  // Phone Number Cell
  Cell := TJSHTMLElement(document.createElement('td'));
  Cell.setAttribute('data-label', 'Phone Number');
  Cell.innerText := PhoneNumber;
  NewRow.appendChild(Cell);

  // Caller Cell
  Cell := TJSHTMLElement(document.createElement('td'));
  Cell.setAttribute('data-label', 'Caller');
  Cell.innerText := Caller;
  NewRow.appendChild(Cell);

  // Time Cell
  Cell := TJSHTMLElement(document.createElement('td'));
  Cell.setAttribute('data-label', 'Time');
  Cell.innerText := DateTimeToStr(IncHour(StrToDateTime(Time), -4));;
  NewRow.appendChild(Cell);

  // Duration Cell
  Cell := TJSHTMLElement(document.createElement('td'));
  Cell.setAttribute('data-label', 'Duration');
  Cell.innerText := Duration;
  NewRow.appendChild(Cell);

  // Transcript Cell
  Cell := TJSHTMLElement(document.createElement('td'));
  Cell.setAttribute('data-label', 'Transcript');
  P := TJSHTMLElement(document.createElement('p'));
  P.className := 'transcript';
  P.innerText := Transcript;
  Cell.appendChild(P);
  NewRow.appendChild(Cell);

  // Listen Button Cell
  Cell := TJSHTMLElement(document.createElement('td'));
  Cell.setAttribute('data-label', 'Listen');

  Button := TJSHTMLElement(document.createElement('button'));
  Button.className := 'btn btn-primary';
  Button.innerHTML := '<i class="fa fa-play"></i>';

  // Open voicemail audio in a new tab on click
  Button.addEventListener('click', procedure(Event: TJSMouseEvent)
  begin
    {$IFNDEF WIN32}
      asm
        var audioElem = document.getElementById('playerAudio');
        audioElem.src = MediaUrl;
        audioElem.load();
        audioElem.play();

        var modal = document.getElementById('playerModal');
        if (modal && modal.parentNode !== document.body) {
          document.body.appendChild(modal);
        }

        // Create the Bootstrap modal object
        var bsModal = new bootstrap.Modal(modal, { keyboard: false });

        // Listen for when the modal *finishes* hiding:
        modal.addEventListener('hidden.bs.modal', function (evt) {
          // Pause and/or reset the audio
          audioElem.pause();
          audioElem.currentTime = 0;
        }, { once: true });

        // Show the modal
        bsModal.show();
      end;
    {$ENDIF}
  end);


  // Add the button to the cell, and the cell to the row
  Cell.appendChild(Button);
  NewRow.appendChild(Cell);

  // Appends new rows to the table body
  TJSHTMLElement(document.getElementById('tblPhoneGrid').getElementsByTagName('tbody')[0]).appendChild(NewRow);
end;

procedure TFViewComplaints.GeneratePagination(TotalPages: Integer);
// Generates pagination for the table.
// TotalPages: Total amount of pages generated by the search
var
  PaginationElement, PageItem, PageLink: TJSHTMLElement;
  I, Start, Finish: Integer;
begin
  PaginationElement := TJSHTMLElement(document.getElementById('pagination'));
  PaginationElement.innerHTML := ''; // Clear existing pagination

  // Previous Button
  PageItem := TJSHTMLElement(document.createElement('li'));
  PageItem.className := 'page-item';
  if PageNumber = 1 then
    PageItem.classList.add('disabled');
  PageLink := TJSHTMLElement(document.createElement('a'));
  PageLink.className := 'page-link';
  PageLink.innerText := 'Previous';
  PageLink.setAttribute('href', 'javascript:void(0)');
  PageLink.addEventListener('click', procedure(Event: TJSMouseEvent)
  begin
    if PageNumber > 1 then
    begin
      Dec(PageNumber);
      GetCalls(GenerateSearchOptions());
    end;
  end);
  PageItem.appendChild(PageLink);
  PaginationElement.appendChild(PageItem);

  // Page Numbers

  if PageNumber <= 4 then
  // If page number is low enough no early elipsis needed
  Begin
    Start := 2;
    Finish := 5;
  End
  else if (PageNumber >= (TotalPages - 3)) then
  // If page number is high enough no late elipsis needed
  begin
    Start := TotalPages - 3;
    Finish := TotalPages - 1;
  end
  else
  begin
    Start := PageNumber - 1;
    Finish := PageNumber + 1;
  end;

  PageItem := TJSHTMLElement(document.createElement('li'));
  PageItem.className := 'page-item';
  if 1 = PageNumber then
    PageItem.classList.add('selected-number'); // Add the selected-number class
  PageLink := TJSHTMLElement(document.createElement('a'));
  PageLink.className := 'page-link';
  PageLink.innerText := '1';
  PageLink.setAttribute('href', 'javascript:void(0)');
  PageLink.addEventListener('click', procedure(Event: TJSMouseEvent)
  var
    PageNum: Integer;
  begin
    PageNum := StrToInt((Event.currentTarget as TJSHTMLElement).innerText);
    PageNumber := PageNum;
    GetCalls(GenerateSearchOptions());
  end);
  PageItem.appendChild(PageLink);
  PaginationElement.appendChild(PageItem);

  // Adds Elipse to pagination if page number is too big
  if PageNumber > 4 then
  begin
    PageItem := TJSHTMLElement(document.createElement('li'));
    PageItem.className := 'page-item';
    PageItem.classList.add('disabled');
    PageLink := TJSHTMLElement(document.createElement('a'));
    PageLink.className := 'page-link';
    PageLink.innerText := '...';
    PageLink.setAttribute('href', 'javascript:void(0)');
    PageItem.appendChild(PageLink);
    PaginationElement.appendChild(PageItem);
  end;


  // Adds Page, page - 1, and page + 1 to pagination
  for I := Start  to Finish do
  begin
    if ( I > 1) and (I < TotalPages) then
    begin
      PageItem := TJSHTMLElement(document.createElement('li'));
      PageItem.className := 'page-item';
      if I = PageNumber then
        PageItem.classList.add('selected-number'); // Add the selected-number class
      PageLink := TJSHTMLElement(document.createElement('a'));
      PageLink.className := 'page-link';
      PageLink.innerText := IntToStr(I);
      PageLink.setAttribute('href', 'javascript:void(0)');
      PageLink.addEventListener('click', procedure(Event: TJSMouseEvent)
      var
        PageNum: Integer;
      begin
        PageNum := StrToInt((Event.currentTarget as TJSHTMLElement).innerText);
        PageNumber := PageNum;
        GetCalls(GenerateSearchOptions());
      end);
      PageItem.appendChild(PageLink);
      PaginationElement.appendChild(PageItem);
    end;
  end;

  // adds ellipse if number is too small
  if PageNumber < TotalPages - 4 then
  begin
    PageItem := TJSHTMLElement(document.createElement('li'));
    PageItem.className := 'page-item';
    PageItem.classList.add('disabled');
    PageLink := TJSHTMLElement(document.createElement('a'));
    PageLink.className := 'page-link';
    PageLink.innerText := '...';
    PageLink.setAttribute('href', 'javascript:void(0)');
    PageItem.appendChild(PageLink);
    PaginationElement.appendChild(PageItem);
  end;

  if TotalPages <> 1 then
  begin
    PageItem := TJSHTMLElement(document.createElement('li'));
    PageItem.className := 'page-item';
    if TotalPages = PageNumber then
          PageItem.classList.add('selected-number');
    PageLink := TJSHTMLElement(document.createElement('a'));
    PageLink.className := 'page-link';
    PageLink.innerText := IntToStr(TotalPages);
    PageLink.setAttribute('href', 'javascript:void(0)');
    PageLink.addEventListener('click', procedure(Event: TJSMouseEvent)
    var
      PageNum: Integer;
    begin
      PageNum := StrToInt((Event.currentTarget as TJSHTMLElement).innerText);
      PageNumber := PageNum;
      GetCalls(generateSearchOptions());
    end);
  end;
  PageItem.appendChild(PageLink);
  PaginationElement.appendChild(PageItem);

  // Next Button
  PageItem := TJSHTMLElement(document.createElement('li'));
  PageItem.className := 'page-item';
  if PageNumber = TotalPages then
    PageItem.classList.add('disabled');
  PageLink := TJSHTMLElement(document.createElement('a'));
  PageLink.className := 'page-link';
  PageLink.innerText := 'Next';
  PageLink.setAttribute('href', 'javascript:void(0)');
  PageLink.addEventListener('click', procedure(Event: TJSMouseEvent)
  begin
    if PageNumber < TotalPages then
    begin
      Inc(PageNumber);
      GetCalls(GenerateSearchOptions());
    end;
  end);
  PageItem.appendChild(PageLink);
  PaginationElement.appendChild(PageItem);

end;


procedure TFViewComplaints.GetCalls(searchOptions: string);
// Retrieves list of calls from the server that meet the criteria searchOptions
// searchOptions: information about how to generate the SQL statement based on
// user input.
var
  xdcResponse: TXDataClientResponse;
  callList : TJSObject;
  i: integer;
  data: TJSArray;
  call: TJSObject;
  callListLength: integer;

begin
  if PageNumber > 0 then
  begin
    Utils.ShowSpinner('spinner');
    xdcResponse := await(XDataWebClient1.RawInvokeAsync('ILookupService.GetCalls',
        [searchOptions]));
    callList :=  TJSObject(xdcResponse.Result);
    data := TJSArray(callList['data']);
    callListLength := integer(callList['count']);
    ClearTable();
    Utils.HideSpinner('spinner');
    for i := 0 to data.Length - 1 do
    begin
      call := TJSObject(data[i]);
      AddRowToTable(string(call['toNumber']), string(call['fromNumber']), string(call['dateCreated']),
      string(call['duration']), string(call['transcription']), string(call['mediaUrl']));
    end;
    TotalPages := (callListLength + PageSize - 1) div PageSize;
    if (PageNumber * PageSize) < callListLength then
    begin
      lblEntries.Caption := 'Showing entries ' + IntToStr((PageNumber - 1) * PageSize + 1) +
                            ' - ' + IntToStr(PageNumber * PageSize) +
                             ' of ' + IntToStr(callListLength);
    end
    else
    begin
       lblEntries.Caption := 'Showing entries ' + IntToStr((PageNumber - 1) * PageSize + 1) +
                            ' - ' + IntToStr(callListLength) +
                             ' of ' + IntToStr(callListLength);
    end;
    GeneratePagination(TotalPages);
  end;
end;


procedure TFViewComplaints.btnApplyClick(Sender: TObject);
// Button that effectively functions as a GetCalls() button
var
searchOptions: string;
begin
  PageNumber := 1;
  PageSize := StrToInt(wcbPageSize.Text);
  StartDate := dtpStartDate.Text;
  EndDate := dtpEndDate.Text;
  OrderBy := wcbSortBy.Text;
  Caller := edtSearch.Text;
  searchOptions := '&phonenumber=' + wcbLocation.Value +
                    '&pagenumber=' + IntToStr(PageNumber) +
                    '&pagesize=' + IntToStr(PageSize) +
                    '&startdate=' + StartDate +
                    '&enddate=' + EndDate +
                    '&orderby=' + OrderBy +
                    '&caller=' + Caller;
  GetCalls(searchOptions);
end;

procedure TFViewComplaints.Search(searchOptions: string);
// Search method that searches the database for a specific phone number
var
  xdcResponse: TXDataClientResponse;
  callList : TJSObject;
  i: integer;
  data: TJSArray;
  call: TJSObject;
  callListLength: integer;

begin
  if PageNumber > 0 then
  begin
    xdcResponse := await(XDataWebClient1.RawInvokeAsync('ILookupService.Search',
        [searchOptions]));
    callList :=  TJSObject(xdcResponse.Result);
    data := TJSArray(callList['data']);
    callListLength := integer(callList['count']);
    ClearTable();
    for i := 0 to data.Length - 1 do
    begin
      call := TJSObject(data[i]);
      AddRowToTable(string(call['toNumber']), string(call['fromNumber']), string(call['dateCreated']),
      string(call['duration']), string(call['transcription']), string(call['mediaUrl']));
    end;
    TotalPages := (callListLength + PageSize - 1) div PageSize;
    lblEntries.Caption := 'Showing entries for phone number: ' + searchOptions;
  end;
end;

procedure TFViewComplaints.btnSearchClick(Sender: TObject);
// calls Search method
begin

end;

procedure TFViewComplaints.ClearTable();
// clears the table
var
  tbody: TJSHTMLElement;
begin
  tbody := TJSHTMLElement(document.getElementById('tblPhoneGrid').getElementsByTagName('tbody')[0]);
  tbody.innerHTML := '';
end;

function TFViewComplaints.GenerateSearchOptions(): string;
// Generates searchOptions for get calls.
var
searchOptions: string;
begin
  PageSize := StrToInt(wcbPageSize.Text);
  StartDate := dtpStartDate.Text;
  EndDate := dtpEndDate.Text;
  OrderBy := wcbSortBy.Text;
  searchOptions := '&phonenumber=' + wcbLocation.Value +
                    '&pagenumber=' + IntToStr(PageNumber) +
                    '&pagesize=' + IntToStr(PageSize) +
                    '&startdate=' + StartDate +
                    '&enddate=' + EndDate +
                    '&orderby=' + OrderBy +
                    '&caller=' + Caller;
  Result := searchOptions;
end;

end.

