unit Common.Config;

interface

const
  defaultServerUrl = 'http://localhost:2009/emiMobile/';

type
  TServerConfig = class
  private
    Furl: string;
    FJWTTokenSecret: string;
    FAdminPassword: string;
    FWebAppFolder: string;
    FReportsFolder: string;
    FMemoLogLevel: Integer;
    FFileLogLevel: Integer;
    FAuditEnabled: Boolean;
  public
    constructor Create;
    property url: string read FUrl write FUrl;
    property jwtTokenSecret: string read FJWTTokenSecret write FJWTTokenSecret;
    property adminPassword: string read FAdminPassword write FAdminPassword;
    property webAppFolder: string read FWebAppFolder write FWebAppFolder;
    property reportsFolder: string read FReportsFolder write FReportsFolder;
    property auditEnabled: Boolean read FAuditEnabled write FAuditEnabled;
    property memoLogLevel: Integer read FMemoLogLevel write FMemoLogLevel;
    property fileLogLevel: Integer read FFileLogLevel write FFileLogLevel;
  end;

procedure LoadServerConfig;

var
  serverConfig: TServerConfig;

implementation

uses
  Bcl.Json, System.SysUtils, System.IOUtils, System.StrUtils,
  Common.Logging;

procedure LoadServerConfig;
var
  configFile: string;
  localConfig: TServerConfig;
begin
  Logger.Log(1, '--LoadServerConfig - start');
  configFile := TPath.ChangeExtension(ParamStr(0), '.json');
  Logger.Log(1, '-- Config file: ' + configFile);

  if TFile.Exists(configFile) then
  begin
    Logger.Log(1, '-- Config file found.');
    localConfig := TJson.Deserialize<TServerConfig>(TFile.ReadAllText(configFile));
    Logger.Log(1, '-- localConfig loaded from config file');
    serverConfig.Free;
    Logger.Log(1, '-- serverConfig.Free - called');
    serverConfig := localConfig;
    Logger.Log(1, '-- serverConfig := localConfig - called');

    Logger.Log(1, '');
    Logger.Log(1, '--- Server Config Values ---');
    Logger.Log(1, '-- url: ' + serverConfig.url + IfThen(serverConfig.url = defaultServerUrl, ' [default]', ' [from config]'));
    Logger.Log(1, '-- adminPassword: ' + serverConfig.adminPassword + IfThen(serverConfig.adminPassword = 'whatisthisusedfor', ' [default]', ' [from config]'));
    Logger.Log(1, '-- jwtTokenSecret: ' + serverConfig.jwtTokenSecret + IfThen(serverConfig.jwtTokenSecret = 'super_secret0123super_secret4567', ' [default]', ' [from config]'));
    Logger.Log(1, '-- webAppFolder: ' + serverConfig.webAppFolder + IfThen(serverConfig.webAppFolder = 'static', ' [default]', ' [from config]'));
    Logger.Log(1, '-- memoLogLevel: ' + IntToStr(serverConfig.memoLogLevel));
    Logger.Log(1, '-- fileLogLevel: ' + IntToStr(serverConfig.fileLogLevel));
    Logger.Log(1, '-- auditEnabled: ' + BoolToStr(serverConfig.auditEnabled, True));
  end
  else
  begin
    Logger.Log(1, '-- Config file not found.');
  end;

  Logger.Log(1, '-------------------------------------------------------------');
  Logger.Log(1, '--LoadServerConfig - end');
end;

{ TServerConfig }

constructor TServerConfig.Create;
begin
  Logger.Log(1, '--TServerConfig.Create - start');
  url := defaultServerUrl;
  adminPassword := 'whatisthisusedfor';
  jwtTokenSecret := 'super_secret0123super_secret4567';
  webAppFolder := 'static';
  reportsFolder := 'reports';
  memoLogLevel := 3;
  fileLogLevel := 4;
  auditEnabled := False;
  Logger.Log(1, '--TServerConfig.Create - end');
end;

end.

