unit Markers;

interface

function SvgPinDataURL(const Hex: string): string;  // complaint: square badge + notch, fill = Hex
function SvgCarDataURL(const Hex: string): string;  // unit: badge + notch with car pictogram, fill = Hex

implementation

uses
  SysUtils;

function URLEncodeSVG(const S: string): string;
var
  R: string;
begin
  R := S;
  R := StringReplace(R, '%',  '%25', [rfReplaceAll]);
  R := StringReplace(R, '#',  '%23', [rfReplaceAll]);
  R := StringReplace(R, '<',  '%3C', [rfReplaceAll]);
  R := StringReplace(R, '>',  '%3E', [rfReplaceAll]);
  R := StringReplace(R, '"',  '%22', [rfReplaceAll]);
  R := StringReplace(R, '''', '%27', [rfReplaceAll]);
  R := StringReplace(R, ' ',  '%20', [rfReplaceAll]);
  Result := 'data:image/svg+xml;charset=utf-8,' + R;
end;

function NormalizeHex(const Hex: string; const Fallback: string): string;
var
  c: string;
begin
  c := Trim(Hex);
  if (Length(c) = 7) and (c[1] = '#') then
    Result := c
  else
    Result := Fallback;
end;

function ForegroundForFill(const Hex: string): string;
var
  c: string;
  r, g, b: Integer;
  lum: Double;
begin
  c := UpperCase(Hex);
  if (Length(c) = 7) and (c[1] = '#') then
  begin
    r := StrToIntDef('$' + Copy(c, 2, 2), 0);
    g := StrToIntDef('$' + Copy(c, 4, 2), 0);
    b := StrToIntDef('$' + Copy(c, 6, 2), 0);
    lum := 0.2126 * r + 0.7152 * g + 0.0722 * b; // 0..255
    if lum > 170 then
      Result := '#111111'
    else
      Result := '#FFFFFF';
  end
  else
    Result := '#FFFFFF';
end;

function SvgPinDataURL(const Hex: string): string;
var
  fill, svg: string;
begin
  fill := NormalizeHex(Hex, '#2563EB');
  svg :=
    '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 30 38">' +
      '<rect x="1" y="1" rx="6" ry="6" width="28" height="28" fill="' + fill + '"/>' +
      '<path d="M15,34 l6,-8 h-12 z" fill="' + fill + '"/>' +
    '</svg>';
  Result := URLEncodeSVG(svg);
end;

function SvgCarDataURL(const Hex: string): string;
var
  fill, fg, svg: string;
begin
  fill := NormalizeHex(Hex, '#2563EB');
  fg   := ForegroundForFill(fill);
  svg :=
    '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 30 38">' +
      '<rect x="1" y="1" rx="6" ry="6" width="28" height="28" fill="' + fill + '"/>' +
      '<path d="M15,34 l6,-8 h-12 z" fill="' + fill + '"/>' +
      // minimalist car pictogram (fits 28x28 body)
      '<path fill="' + fg + '" d="M7 20h16l-2-6a3 3 0 0 0-2.8-2H11.8A3 3 0 0 0 9 14l-2 6zm-1 1h-1a1 1 0 0 0 0 2h1v2a1 1 0 0 0 1 1h2a1 1 0 0 0 1-1v-2h8v2a1 1 0 0 0 1 1h2a1 1 0 0 0 1-1v-2h1a1 1 0 0 0 0-2h-1zm4-7h10l1 3H9l1-3zM10 22a2 2 0 1 0 0 4 2 2 0 0 0 0-4zm10 0a2 2 0 1 0 0 4 2 2 0 0 0 0-4z"/>' +
    '</svg>';
  Result := URLEncodeSVG(svg);
end;

end.

