program emiMobileServer;

uses
  FastMM4,
  System.SyncObjs,
  System.SysUtils,
  Vcl.StdCtrls,
  IniFiles,
  Vcl.Forms,
  Api.Server.Module in 'Source\Api.Server.Module.pas' {ApiServerModule: TDataModule},
  Main in 'Source\Main.pas' {FMain},
  Common.Logging in 'Source\Common.Logging.pas',
  Api.Database in 'Source\Api.Database.pas' {ApiDatabaseModule: TDataModule},
  Common.Middleware.Logging in 'Source\Common.Middleware.Logging.pas',
  Common.Config in 'Source\Common.Config.pas',
  Auth.Server.Module in 'Source\Auth.Server.Module.pas' {AuthServerModule: TDataModule},
  Auth.Database in 'Source\Auth.Database.pas' {AuthDatabase: TDataModule},
  uLibrary in 'Source\uLibrary.pas',
  Auth.Service in 'Source\Auth.Service.pas',
  Api.Service in 'Source\Api.Service.pas',
  Auth.ServiceImpl in 'Source\Auth.ServiceImpl.pas',
  Api.ServiceImpl in 'Source\Api.ServiceImpl.pas',
  App.Server.Module in 'Source\App.Server.Module.pas' {AppServerModule: TDataModule},
  Common.Ini in 'Source\Common.Ini.pas';

type
  TMemoLogAppender = class( TInterfacedObject, ILogAppender )
  private
    FLogLevel: Integer;
    FLogMemo: TMemo;
    FCriticalSection: TCriticalSection;
  public
    constructor Create(ALogLevel: Integer; ALogMemo: TMemo);
    destructor Destroy; override;
    procedure Send(logLevel: Integer; Log: ILog);
  end;

  TFileLogAppender = class( TInterfacedObject, ILogAppender )
  private
    FLogLevel: Integer;
    FFilename: string;
    FLogDirectory: string;
    FCriticalSection: TCriticalSection;
    function GetLogFilePath: string;
  public
    constructor Create(ALogLevel: Integer; AFilename: string);
    destructor Destroy; override;
    procedure Send(logLevel: Integer; Log: ILog);
  end;

{ TMemoLogAppender }
constructor TMemoLogAppender.Create(ALogLevel: Integer; ALogMemo: TMemo);
begin
  FLogLevel := ALogLevel;
  FLogMemo := ALogMemo;
  FCriticalSection := TCriticalSection.Create;
end;

destructor TMemoLogAppender.Destroy;
begin
  FCriticalSection.Free;
  inherited;
end;

procedure TMemoLogAppender.Send(logLevel: Integer; Log: ILog);
var
  FormattedMessage: string;
  LogTime: TDateTime;
  LogMsg: string;

begin
  FCriticalSection.Acquire;
  try
    LogTime := Now;

    FormattedMessage := FormatDateTime('[yyyy-mm-dd HH:nn:ss.zzz]', LogTime);
    LogMsg := Log.GetMessage;
    if LogMsg.IsEmpty then
      FormattedMessage := ''
    else
      FormattedMessage := FormattedMessage + '[' + IntToStr(logLevel) +'] ' + LogMsg;

    if logLevel <= FLogLevel then
      FLogMemo.Lines.Add( FormattedMessage );
  finally
    FCriticalSection.Release;
  end;
end;

{ TFileLogAppender }
constructor TFileLogAppender.Create(ALogLevel: integer; AFilename: string);
var
  iniFile: TIniFile;
  fileNum: integer;
  logsDir: string;
begin
  FLogLevel := ALogLevel;
  FCriticalSection := TCriticalSection.Create;
  FLogDirectory := ExtractFilePath(Application.ExeName) + 'logs\';
  if not DirectoryExists(FLogDirectory) then
    CreateDir(FLogDirectory);

  iniFile := TIniFile.Create( ChangeFileExt(Application.ExeName, '.ini') );
  try
    fileNum := iniFile.ReadInteger( 'Settings', 'LogFileNum', 0 );
    FFilename := AFilename + Format( '%.4d', [fileNum] );
    iniFile.WriteInteger( 'Settings', 'LogFileNum', fileNum + 1 );
  finally
    iniFile.Free;
  end;
end;

destructor TFileLogAppender.Destroy;
begin
  FCriticalSection.Free;
  inherited;
end;

function TFileLogAppender.GetLogFilePath: string;
begin
  Result := FLogDirectory + FFilename + '.log';
end;


procedure TFileLogAppender.Send(logLevel: integer; Log: ILog);
var
  FormattedMessage: string;
  LogFile: string;
  LogTime: TDateTime;
  LogMsg: string;
  FLogFile: TextFile;

begin
  FCriticalSection.Acquire;
  try
    LogTime := Now;
    LogFile := GetLogFilePath;

    formattedMessage := FormatDateTime('[yyyy-mm-dd HH:nn:ss.zzz]', logTime);
    logMsg := Log.GetMessage;
    if logMsg.IsEmpty then
      formattedMessage := ''
    else
      formattedMessage := formattedMessage + '[' + IntToStr(logLevel) +'] ' + logMsg;

    try
      AssignFile( FLogFile, LogFile );
      if FileExists(LogFile) then
        Append( FLogFile )
      else
        ReWrite( FLogFile );
      if logLevel <= FLogLevel then
        WriteLn( FLogFile, FormattedMessage );
    finally
      CloseFile(FLogFile);
    end;
  finally
    FCriticalSection.Release;
  end;
end;

{$R *.res}

var
  iniFile: TIniFile;
  memoLogLevel: Integer;
  fileLogLevel: Integer;

begin
  ReportMemoryLeaksOnShutdown := True;
  Application.Initialize;
  Application.MainFormOnTaskbar := True;
  Application.CreateForm(TFMain, FMain);
  iniFile := TIniFile.Create( ChangeFileExt(Application.ExeName, '.ini') );
  try
    memoLogLevel := IniFile.ReadInteger( 'Settings', 'MemoLogLevel', 3 );
    fileLogLevel := IniFile.ReadInteger( 'Settings', 'FileLogLevel', 4 );
  finally
    iniFile.Free;
  end;
  Logger.AddAppender(TMemoLogAppender.Create( memoLogLevel, FMain.memoinfo ));
  Logger.AddAppender(TFileLogAppender.Create( fileLogLevel, 'webPoliceReports' ));
  Application.Run;
end.
