unit View.RequestDemo;

interface

uses
  System.SysUtils, System.Classes, JS, Web, WEBLib.Graphics, WEBLib.Controls,
  WEBLib.Forms, WEBLib.Dialogs, Vcl.StdCtrls, WEBLib.StdCtrls, Vcl.Controls,
  XData.Web.Client, ConnectionModule, Utils, WEBLib.RegularExpressions;

type
  TFRequestDemo = class(TWebForm)
    edtReqName: TWebEdit;
    edtReqEmail: TWebEdit;
    edtReqAgency: TWebEdit;
    edtReqPhoneNumber: TWebEdit;
    lblReqPrefMethod: TWebLabel;
    radReqPhone: TWebRadioButton;
    radReqEmail: TWebRadioButton;
    btnReqSendRequest: TWebButton;
    xdwcRequestDemo: TXDataWebClient;
    procedure btnReqSendRequestClick(Sender: TObject);
    procedure WebFormCreate(Sender: TObject);
    procedure xdwcRequestDemoError(Error: TXDataClientError);
  private
    function IsEmailValid(const email: string): Boolean;
    function PreferredMethod: string;
    procedure NavScrollSizing;
    procedure ClearFieldErrors;
    procedure SetInvalid(const inputId: string);
    procedure ShowFeedback(const feedbackId: string);
    function ValidateRequest: Boolean;
    procedure SendRequest;
  end;

var
  FRequestDemo: TFRequestDemo;

implementation

{$R *.dfm}

uses
  Site.Navbar,
  Site.Footer;

procedure TFRequestDemo.WebFormCreate(Sender: TObject);
begin
  Application.ThemeColor := clTMSWEB;
  InjectNavbar;
  InjectFooter;
  SetFooterVersion('ver - ' + TDMConnection.clientVersion);
  NavScrollSizing;
end;

procedure TFRequestDemo.xdwcRequestDemoError(Error: TXDataClientError);
begin
  Utils.HideSpinner('spinner');
  ShowNotificationModal('Error when attempting to send: ' + Error.ErrorMessage);
end;

procedure TFRequestDemo.NavScrollSizing;
begin
  asm
    window.addEventListener('scroll', function() {
      var navbar = document.querySelector('.em-navbar-wrap .navbar');
      var logo = document.querySelector('.em-brand img');
      var scrollDistance = 50;

      if (window.scrollY > scrollDistance) {
        navbar.classList.add('scrolled');
        logo.classList.add('scrolled-logo');
      } else {
        navbar.classList.remove('scrolled');
        logo.classList.remove('scrolled-logo');
      }
    });
  end;
end;

function TFRequestDemo.IsEmailValid(const email: string): Boolean;
const
  pattern = '^[^\s@]+@[^\s@]+\.[^\s@]+$';
begin
  Result := TRegEx.IsMatch(email.Trim, pattern);
end;

function TFRequestDemo.PreferredMethod: string;
begin
  if radReqPhone.Checked then
    Exit('phone');
  if radReqEmail.Checked then
    Exit('email');
  Result := '';
end;

procedure TFRequestDemo.ShowFeedback(const feedbackId: string);
var
  el: TJSHTMLElement;
begin
  el := TJSHTMLElement(document.getElementById(feedbackId));
  if Assigned(el) then
    el.classList.remove('d-none');
end;

procedure TFRequestDemo.SetInvalid(const inputId: string);
var
  inputEl: TJSHTMLElement;
begin
  inputEl := TJSHTMLElement(document.getElementById(inputId));
  if Assigned(inputEl) then
    inputEl.classList.add('is-invalid');
end;

procedure TFRequestDemo.ClearFieldErrors;
var
  inputEl: TJSHTMLElement;
  prefEl: TJSHTMLElement;
begin
  inputEl := TJSHTMLElement(document.getElementById('edt_req_name'));
  if Assigned(inputEl) then inputEl.classList.remove('is-invalid');

  inputEl := TJSHTMLElement(document.getElementById('edt_req_email'));
  if Assigned(inputEl) then inputEl.classList.remove('is-invalid');

  inputEl := TJSHTMLElement(document.getElementById('edt_req_phone_number'));
  if Assigned(inputEl) then inputEl.classList.remove('is-invalid');

  prefEl := TJSHTMLElement(document.getElementById('fb_req_pref'));
  if Assigned(prefEl) then
    prefEl.classList.add('d-none');
end;


function TFRequestDemo.ValidateRequest: Boolean;
var
  prefMissing: Boolean;
  needsEmail: Boolean;
  needsPhone: Boolean;
  emailHasText: Boolean;
begin
  ClearFieldErrors;

  if edtReqName.Text.Trim = '' then
  begin
    SetInvalid('edt_req_name');
    Exit(False);
  end;

  prefMissing := not (radReqPhone.Checked or radReqEmail.Checked);
  if prefMissing then
  begin
    ShowFeedback('fb_req_pref');
    Exit(False);
  end;

  needsEmail := radReqEmail.Checked;
  needsPhone := radReqPhone.Checked;

  emailHasText := edtReqEmail.Text.Trim <> '';

  if (needsEmail or emailHasText) and (not IsEmailValid(edtReqEmail.Text)) then
  begin
    SetInvalid('edt_req_email');
    Exit(False);
  end;

  if needsPhone and (edtReqPhoneNumber.Text.Trim = '') then
  begin
    SetInvalid('edt_req_phone_number');
    Exit(False);
  end;

  Result := True;
end;


procedure TFRequestDemo.btnReqSendRequestClick(Sender: TObject);
begin
  if not ValidateRequest then
    Exit;

  Utils.ShowSpinner('spinner');

  if not DMConnection.ApiConnection.Connected then
  begin
    DMConnection.ApiConnection.Open(
      procedure
      begin
        SendRequest;
      end);
    Exit;
  end;

  SendRequest;
end;

procedure TFRequestDemo.SendRequest;
var
  bodyText: string;
begin
  bodyText :=
    'Demo request details:' + sLineBreak +
    'Name: ' + edtReqName.Text + sLineBreak +
    'Email: ' + edtReqEmail.Text + sLineBreak +
    'Phone: ' + edtReqPhoneNumber.Text + sLineBreak +
    'Agency: ' + edtReqAgency.Text + sLineBreak +
    'Preferred contact: ' + PreferredMethod;

  xdwcRequestDemo.RawInvoke('IApiService.SendEmail',
    [edtReqName.Text, edtReqEmail.Text, 'Demo Request', bodyText],
    procedure(response: TXDataClientResponse)
    begin
      Utils.HideSpinner('spinner');
      ClearFieldErrors;

      edtReqName.Text := '';
      edtReqEmail.Text := '';
      edtReqPhoneNumber.Text := '';
      edtReqAgency.Text := '';
      radReqPhone.Checked := False;
      radReqEmail.Checked := False;

      ShowNotificationModal('Your demo request has been sent successfully. We will reach out shortly.');
    end);
end;

initialization
  RegisterClass(TFRequestDemo);

end.

